!===============================================================================
! Copyright (C) 2021 Intel Corporation
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

! Content:
!       Example of using dfftw_plan_many_dft function on a
!       (GPU) device using the OpenMP target (offload) interface.
!
!*****************************************************************************

include "fftw/offload/fftw3_omp_offload.f90"

program dp_plan_many_dft_2d

  use FFTW3_OMP_OFFLOAD
  use omp_lib, ONLY : omp_get_num_devices
  use, intrinsic :: ISO_C_BINDING

  include 'fftw/fftw3.f'

  ! Sizes of 2D transform and the number of them
  integer, parameter :: N1 = 8
  integer, parameter :: N2 = 16
  integer, parameter :: M = 3

  ! Arbitrary harmonic used to verify FFT
  integer, parameter :: H1 = 1
  integer, parameter :: H2 = -N2/2

  ! Need double precision
  integer, parameter :: WP = selected_real_kind(15,307)

  ! Execution status
  integer :: statusf = 0, statusb = 0, status = 0

  ! Data arrays
  complex(WP), allocatable :: x(:)

  ! Distance, stride and embedding array
  integer :: dist, stride, nembed(2)

  ! FFTW plan
  integer*8 :: fwd = 0, bwd = 0

  stride = 1
  nembed(1) = stride * N1
  nembed(2) = N2
  dist = nembed(1) * nembed(2)

  print *,"Example dp_plan_many_dft_2d"
  print *,"Forward and backward multiple 2D complex in-place FFT"
  print *,"Configuration parameters:"
  print '("  N = ["I0","I0"]")', N1, N2
  print '("  H = ["I0","I0"]")', H1, H2
  print '("  nembed = ["I0","I0"]")', nembed(1), nembed(2)
  print '("  M = "I0)', M
  print '("  dist = "I0)', dist
  print '("  stride = "I0)', stride


  print *,"Allocate data array"
  allocate ( x(dist * M), STAT = status)
  if (0 /= status) goto 999

  print *,"Initialize data for forward transform"
  call init(x, N1, N2, H1, H2, M, dist)

  print *,"Create FFTW forward transform plan"
  !$omp target data map(tofrom:x)
  !$omp dispatch
  call dfftw_plan_many_dft(fwd, 2, (/N1, N2/), M, &
                           x, nembed, stride, dist,   &
                           x, nembed, stride, dist,   &
                           FFTW_FORWARD, FFTW_ESTIMATE)
  if (0 == fwd) print *, "Call to dfftw_plan_many_dft for forward transform &
                          &has failed"

  print *,"Create FFTW backward transform plan"
  !$omp dispatch
  call dfftw_plan_many_dft(bwd, 2, (/N1, N2/), M, &
                           x, nembed, stride, dist,   &
                           x, nembed, stride, dist,   &
                           FFTW_BACKWARD, FFTW_ESTIMATE)
  if (0 == bwd) print *, "Call to dfftw_plan_many_dft for backward transform &
                          &has failed"

  print *,"Compute forward transform"
  !$omp dispatch
  call dfftw_execute_dft(fwd, x, x)

  ! Update the host with the results from forward FFT
  !$omp target update from(x)

  print *,"Verify the result of the forward transform"
  statusf = verify(x, N1, N2, H1, H2, M, dist)

  print *,"Initialize data for backward transform"
  call init(x, N1, N2, -H1, -H2, M,dist)

  ! Update the device with input for backward FFT
  !$omp target update to(x)

  print *,"Compute backward transform"
  !$omp dispatch
  call dfftw_execute_dft(bwd, x, x)
  !$omp end target data

  print *,"Verify the result of the backward transform"
  statusb = verify(x, N1, N2, H1, H2, M, dist)
  if ((0 /= statusf) .or. (0 /= statusb)) goto 999

100 continue

  print *,"Destroy FFTW plans"
  call dfftw_destroy_plan(fwd)
  call dfftw_destroy_plan(bwd)

  print *,"Deallocate arrays"
  deallocate(x)

  if (status == 0) then
    print *, "TEST PASSED"
    call exit(0)
  else
    print *, "TEST FAILED"
    call exit(1)
  end if

999 print '("  Error, status forward = ",I0)', statusf
  print '(" Error, status backward = ",I0)', statusb
  status = 1
  goto 100

contains

  ! Compute mod(K*L,M) accurately
  pure integer*8 function moda(k,l,m)
    integer, intent(in) :: k,l,m
    integer*8 :: k8
    k8 = k
    moda = mod(k8*l,m)
  end function moda

  ! Initialize array with harmonic /H1, H2/
  subroutine init(x, N1, N2, H1, H2, M, dist)
    integer N1, N2, H1, H2, M, dist
    complex(WP) :: x(:)

    integer k1, k2, k3
    complex(WP), parameter :: I_TWOPI = (0.0_WP,6.2831853071795864769_WP)

    do k1 = 1, M
      do k2 = 1, N1
        do k3 = 1, N2
          x(((k1-1)*dist)+(((k3-1)*N1)+k2)) = exp ( I_TWOPI * ( &
          real(  moda(k2-1,H1, N1), WP)/N1 &
          + real(moda(k3-1,H2, N2), WP)/N2 )) / (N1*N2)
        end do
      end do
    end do
  end subroutine init

  ! Verify that x(N1,N2) is unit peak at x(H1,H2)
  integer function verify(x, N1, N2, H1, H2, M, dist)
    integer N1, N2, H1, H2, M, dist
    complex(WP) :: x(:)

    integer k1, k2, k3
    real(WP) err, errthr, maxerr
    complex(WP) :: res_exp, res_got

    ! Note, this simple error bound doesn't take into account error of
    ! input data
    errthr = 5.0 * log(real(N1*N2,WP)) / log(2.0_WP) * EPSILON(1.0_WP)
    print '("  Check if err is below errthr " G10.3)', errthr

    maxerr = 0.0_WP
    do k3 = 1, M
      do k2 = 1, N2
        do k1 = 1, N1
          if (mod(k1-1-H1, N1)==0 .AND. mod(k2-1-H2, N2)==0) then
            res_exp = 1.0_WP
          else
            res_exp = 0.0_WP
          end if
          res_got = x(((k3-1)*dist)+(k2-1)*N1+k1)
          err = abs(res_got - res_exp)
          maxerr = max(err,maxerr)
          if (.not.(err < errthr)) then
            print '("  x("I0","I0"):"$)', k1, k2
            print '(" expected ("G24.17","G24.17"),"$)', res_exp
            print '(" got ("G24.17","G24.17"),"$)', res_got
            print '(" err "G10.3)', err
            print *," Verification FAILED"
            verify = 1
            return
          end if
        end do
      end do
    end do
    print '("  Verified,  maximum error was " G10.3)', maxerr
    verify = 0
  end function verify
end program dp_plan_many_dft_2d
