/*******************************************************************************
* Copyright (C) 2025 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*    Content : Intel(R) oneAPI Math Kernel Library (oneMKL)
*              Inspector-Executor Sparse BLAS Fortran example
*              for the mkl_sparse_?_dense2bsr routine to demonstrate
*              conversion from a dense matrix to a block compressed
*              sparse row (BSR) sparse matrix format.
*
********************************************************************************
*
* Example program for using Intel(R) oneMKL Inspector-Executor Sparse BLAS
* routines for conversions from dense to sparse matrix representation.
*
* The following Inspector-Executor Sparse BLAS routines are used in the example:
*
*   Initialization/Destruction stage:
*          mkl_sparse_destroy
*
*   Format conversion function:
*          mkl_sparse_d_dense2bsr
*
*   Export function for sparse data arrays:
*          mkl_sparse_d_export_bsr
*
* Consider the matrix A below to be represented in block compressed sparse row
* (BSR) format (see 'Sparse Matrix Storage Schemes' in the Intel(R) oneMKL
* Fortran Reference Manual):
*
*       |  1  -2   0   0 |
*       |  3  -4   0   0 |
*   A = |  0   0   5  -6 |.
*       |  0   0   7  -8 |
*       |  9 -10   0   0 |
*       | 11 -12   0   0 |
*
*
* A BSR representation of the matrix with row-major layout, 0-based indexing,
* block size 2 (i.e. with square blocks of dimension 2x2), and three arrays is:
*
*     bsrNrows       = 3
*     bsrNcols       = 2
*     bsrBlockLayout = SPARSE_LAYOUT_ROW_MAJOR
*     bsrBlockSize   = 2
*     bsrNnz         = 3
*     bsrIndex       = SPARSE_INDEX_BASE_ZERO
*     bsrRowPtr      = (0               1               2               3)
*     bsrColIdx      = (0               1               0)
*     bsrValues      = (1  -2   3  -4   5  -6   7  -8   9 -10  11 -12)
*
* This example presents:
*    * mkl_sparse_d_dense2bsr() usage to convert a dense matrix to
*      a sparse matrix in BSR format
*    * manual computation of a reference dense matrix-vector product from the
*      original dense matrix
*    * usage of mkl_sparse_d_mv() for the computation of a sparse matrix-vector
*      product using the sparse matrix in BSR format
*    * comparison and validation of the resulting vectors
*
*
* Note: The I/E Sparse BLAS solution supports conversion from a dense matrix to
*       any of the following sparse formats:
*       - Compressed Sparse Row (CSR)
*       - Compressed Sparse Column (CSC)
*       - Block Compressed Sparse Row (BSR)
*       - Coordinate (COO)
*       This example demonstrates conversion to BSR format using the
*       mkl_sparse_?_dense2bsr() API. To convert to other sparse formats,
*       follow a similar approach and use the corresponding API:
*       - mkl_sparse_?_dense2csr()
*       - mkl_sparse_?_dense2csc()
*       - mkl_sparse_?_dense2coo()
*
********************************************************************************
*/
#include <stdio.h>
#include <assert.h>
#include <math.h>
#include "mkl.h"

#ifdef MKL_ILP64
#define INT_PRINT_FORMAT "%4lld"
#else
#define INT_PRINT_FORMAT "%4d"
#endif

void print_int_value(const char *name, MKL_INT val) {
    printf("\t\t%s = " INT_PRINT_FORMAT "\n", name, val);
}

void print_int_array(const char *name, MKL_INT *array, MKL_INT len) {
    printf("\t\t%s =", name);
    for (MKL_INT i = 0; i < len; ++i) {
        printf(INT_PRINT_FORMAT ",", array[i]);
    }
    printf("\n");
}

void print_index(const char *name, sparse_index_base_t idxBase) {
    printf("\t\t%s = %s\n", name, idxBase == SPARSE_INDEX_BASE_ZERO ? "SPARSE_INDEX_BASE_ZERO" : "SPARSE_INDEX_BASE_ONE");
}

void print_layout(const char *name, sparse_layout_t layout) {
    printf("\t\t%s = %s\n", name, layout == SPARSE_LAYOUT_ROW_MAJOR ? "SPARSE_LAYOUT_ROW_MAJOR" : "SPARSE_LAYOUT_COLUMN_MAJOR");
}

void print_flt_array(const char *name, double *array, MKL_INT len) {
    printf("\t\t%s =", name);
    for (MKL_INT i = 0; i < len; ++i) {
        printf("%4.0f,", array[i]);
    }
    printf("\n");
}

int main() {
    //*******************************************************************************
    //     Declaration and initialization of parameters for the dense representation
    //     of the matrix A
    //*******************************************************************************
#define M 6
#define N 4

    sparse_layout_t dense_layout = SPARSE_LAYOUT_ROW_MAJOR;
    MKL_INT ldA = N;
    double denseMatrix[M * N] = {  1,  -2,   0,   0,
                                   3,  -4,   0,   0,
                                   0,   0,   5,  -6,
                                   0,   0,   7,  -8,
                                   9, -10,   0,   0,
                                  11, -12,   0,   0 };


    //*******************************************************************************
    //     Declaration and initialization of parameters for the sparse representation
    //     of the matrix A in BSR format
    //*******************************************************************************
    sparse_matrix_t bsrA = NULL;
    struct matrix_descr descrA;
    descrA.type = SPARSE_MATRIX_TYPE_GENERAL;
    sparse_index_base_t indexing = SPARSE_INDEX_BASE_ZERO;
    sparse_layout_t bsrBlockLayout = SPARSE_LAYOUT_ROW_MAJOR;
    MKL_INT bsrBlockSize = 2;
    sparse_status_t status;
    int exit_status = 0;

    //*******************************************************************************
    //     Declaration and initialization of dense vectors
    //*******************************************************************************
    double x[N]     = {1., 1., 1., 1.};
    double y_ref[M] = {0., 0., 0., 0., 0., 0.};
    double y[M]     = {0., 0., 0., 0., 0., 0.};

    printf( "\nExample program for conversion from a dense matrix to\n" );
    printf( " a sparse matrix in BSR format using IE Sparse BLAS APIs\n" );
    printf( "-------------------------------------------------------------------------\n" );

    //******************************************************************************
    //    Input dense matrix
    //******************************************************************************
    printf("\t[Input] Matrix array in dense format:\n");
    print_int_value("nrows", M);
    print_int_value("ncols", N);
    printf("\t\tdenseMatrix:\n");
    for (MKL_INT i = 0; i < M; ++i) {
        printf("\t\t");
        for (MKL_INT j = 0; j < N; ++j) {
            printf(" %6.0lf ", denseMatrix[i * N + j]);
        }
        printf("\n");
    }
    printf("\n");

    //******************************************************************************
    //    Convert from DENSE to BSR format
    //******************************************************************************
    status = mkl_sparse_d_dense2bsr(M, N, dense_layout, ldA, denseMatrix,
                                    indexing, bsrBlockSize, bsrBlockLayout, descrA, &bsrA);

    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_dense2bsr: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //******************************************************************************
    //    Export and print the BSR matrix
    //******************************************************************************
    sparse_index_base_t indexingOut;
    sparse_layout_t bsrBlockLayoutOut;
    MKL_INT bsrBlockNrows, bsrBlockNcols, bsrBlockSizeOut;
    MKL_INT *bsrRowStart, *bsrRowEnd, *bsrColIdx;
    double *bsrValues;
    status = mkl_sparse_d_export_bsr(bsrA, &indexingOut, &bsrBlockLayoutOut, &bsrBlockNrows,
                                     &bsrBlockNcols, &bsrBlockSizeOut, &bsrRowStart,
                                     &bsrRowEnd, &bsrColIdx, &bsrValues);
    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_export_bsr: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    // Check if output BSR parameters match input ones
    if (bsrBlockLayoutOut != bsrBlockLayout) {
        printf("\t Error: blockLayoutOut does not match input bsrBlockLayout\n");
        exit_status = 1;
        goto exit;
    }
    if (bsrBlockSizeOut != bsrBlockSize) {
        printf("\t Error: blockSizeOut does not match input bsrBlockSize\n");\
        exit_status = 1;
        goto exit;
    }
    if (indexingOut != indexing) {
        printf("\t Error: indexingOut does not match input indexing\n");
        exit_status = 1;
        goto exit;
    }

    printf("\t[Output] Matrix in BSR format:\n");
    MKL_INT index = indexing == SPARSE_INDEX_BASE_ZERO ? 0 : 1;
    MKL_INT bsrBlockNnz = bsrRowEnd[bsrBlockNrows - 1] - index;
    MKL_INT bsrNnz = bsrBlockNnz * bsrBlockSize * bsrBlockSize;
    print_int_value("bsrNrows", bsrBlockNrows);
    print_int_value("bsrNcols", bsrBlockNcols);
    print_int_value("bsrBlockSize", bsrBlockSize);
    print_layout("bsrBlockLayout", bsrBlockLayout);
    print_index("bsrIndexing", indexing);
    print_int_value("bsrBlockNnz", bsrBlockNnz);
    print_int_value("bsrNnz", bsrNnz);
    print_int_array("bsrRowPtr", bsrRowStart, bsrBlockNrows + 1);
    print_int_array("bsrColIdx", bsrColIdx, bsrBlockNnz);
    print_flt_array("bsrValues", bsrValues, bsrNnz);
    printf("\n");

    //******************************************************************************
    //    Compute the reference matrix-vector solution from the dense matrix
    //******************************************************************************
    for (MKL_INT i = 0; i < M; ++i) {
        y_ref[i] = 0;
        for (MKL_INT j = 0; j < N; ++j) {
            y_ref[i] += denseMatrix[i * N + j] * x[j];
        }
    }

    //******************************************************************************
    //    Compute the matrix-vector solution using the sparse matrix in BSR format
    //******************************************************************************
    status = mkl_sparse_d_mv(SPARSE_OPERATION_NON_TRANSPOSE, 1.0, bsrA, descrA, x, 0.0, y);

    if (status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_d_mv: %d \n", status);
        exit_status = 1;
        goto exit;
    }

    //******************************************************************************
    //    Validate the resulting vector y against the reference solution y_ref
    //******************************************************************************
    int conversion_passed = 1;
    double TOL = 1.0e-14;
    for (MKL_INT i = 0; i < M; ++i) {
        if (fabs(y[i] - y_ref[i]) > TOL) {
            conversion_passed = 0;
            break;
        }
    }

    if (conversion_passed) {
        printf("\t The conversion from dense representation to BSR format passed\n");
    }
    else
    {
        printf("\t The conversion from dense representation to BSR format failed\n");
        exit_status = 1;
        goto exit;
    }
    printf("\n");

exit:
    // Release matrix handle
    if (bsrA) mkl_sparse_destroy(bsrA);

    return exit_status;
}
