/*******************************************************************************
* Copyright (C) 2019 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP API oneapi::mkl::sparse::trsv to perform
*       sparse triangular solve on a SYCL device (CPU, GPU). This example uses a
*       sparse matrix in CSR format.
*
*       op(A) * x = alpha * b
*
*       where op() is defined by one of
*           oneapi::mkl::transpose::{nontrans,trans,conjtrans}
*
*       Currently, only op() = nontrans is supported for TRSV
*
*       The supported floating point data types for trsv are:
*           float
*           double
*           std::complex<float>
*           std::complex<double>
*
*       The supported matrix formats for trsv are:
*           CSR
*           COO (currently only on CPU device)
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

// local includes
#include "common_for_examples.hpp"
#include "./include/common_for_sparse_examples.hpp"

//
// Main example for Sparse Triangular Solver consisting of
// initialization of A matrix, b and x vectors.
// Then the following system is solved
//
// op(A) * x = alpha * b
//
// and finally the results are post processed.
//
template <typename dataType, typename intType>
int run_sparse_blas_example(sycl::queue &q)
{

    bool good = true;

    //
    // handle for sparse matrix
    //
    oneapi::mkl::sparse::matrix_handle_t csrA = nullptr;

    try {

        // Initialize data for Sparse Triangular Solve
        oneapi::mkl::uplo uplo_val             = oneapi::mkl::uplo::lower;
        oneapi::mkl::diag diag_val             = oneapi::mkl::diag::nonunit;
        oneapi::mkl::index_base index_base_val = oneapi::mkl::index_base::zero;
        if constexpr (is_complex<dataType>()) {
            uplo_val       = oneapi::mkl::uplo::upper;
            diag_val       = oneapi::mkl::diag::unit;
            index_base_val = oneapi::mkl::index_base::one;
        }
        intType a_ind = (index_base_val == oneapi::mkl::index_base::zero ? 0 : 1);

        oneapi::mkl::transpose transA = oneapi::mkl::transpose::nontrans;

        dataType alpha;
        if constexpr (!is_complex<dataType>()) {
            alpha = set_fp_value(dataType(2.0), dataType(0.0));
        }
        else {
            alpha = set_fp_value(dataType(2.0), dataType(-1.0));
        }

        // Matrix data size
        intType size  = 4;
        const std::int64_t nrows = size * size * size;
        const std::int64_t ncols = nrows;

        // Input matrix in CSR format
        std::vector<intType, mkl_allocator<intType, 64>> ia;
        std::vector<intType, mkl_allocator<intType, 64>> ja;
        std::vector<dataType, mkl_allocator<dataType, 64>> a;

        ia.resize(nrows + 1);
        ja.resize(27 * nrows);
        a.resize(27 * nrows);

        generate_sparse_matrix<dataType, intType>(size, ia, ja, a, a_ind);

        const std::int64_t nnz = ia[nrows] - a_ind;

        // Vectors b and x
        std::vector<dataType, mkl_allocator<dataType, 64>> b;
        std::vector<dataType, mkl_allocator<dataType, 64>> x;
        std::vector<dataType, mkl_allocator<dataType, 64>> x_ref;
        b.resize(nrows);
        x.resize(nrows);
        x_ref.resize(nrows);

        // Init vectors b and x
        for (int i = 0; i < nrows; i++) {
            b[i] = set_fp_value(dataType(1.0), dataType(0.0));
            x[i] = set_fp_value(dataType(0.0), dataType(0.0));
            x_ref[i] = set_fp_value(dataType(0.0), dataType(0.0));
        }

        sycl::buffer<intType, 1> ia_buffer(ia.data(), ia.data() + nrows + 1);
        sycl::buffer<intType, 1> ja_buffer(ja.data(), ja.data() + nnz);
        sycl::buffer<dataType, 1> a_buffer(a.data(), a.data() + nnz);
        sycl::buffer<dataType, 1> b_buffer(b.data(), b.data() + b.size());
        sycl::buffer<dataType, 1> x_buffer(x.data(), x.data() + x.size());

        //
        // Execute Triangular Solve
        //
        std::cout << "\n\t\tsparse::trsv parameters: Solve for x in op(uplo(A)) * x = alpha * b\n";
        std::cout << "\t\t\ttransA   = " << transA << std::endl;
        std::cout << "\t\t\tindexing = " << index_base_val << std::endl;
        std::cout << "\t\t\tuplo     = " << uplo_val << std::endl;
        std::cout << "\t\t\tdiag     = " << diag_val << std::endl;
        std::cout << "\t\t\talpha    = " << alpha << std::endl;
        std::cout << "\t\t\tnrows    = " << nrows << std::endl;

        oneapi::mkl::sparse::init_matrix_handle(&csrA);

        oneapi::mkl::sparse::set_csr_data(q, csrA, nrows, ncols, nnz, index_base_val,
                                          ia_buffer, ja_buffer, a_buffer);

        oneapi::mkl::sparse::optimize_trsv(q, uplo_val, transA, diag_val, csrA);

        // add oneapi::mkl::sparse::trsv to execution queue
        oneapi::mkl::sparse::trsv(q, uplo_val, transA, diag_val, alpha, csrA,
                                  b_buffer, x_buffer);

        oneapi::mkl::sparse::release_matrix_handle(q, &csrA);

        q.wait();

        //
        // Post Processing
        //

        auto res = x_buffer.get_host_access(sycl::read_only);

        // Solve for x_ref in  op(uplo(A)) * x_ref = b  with op() == nontrans
        if (uplo_val == oneapi::mkl::uplo::lower) {
            for (intType row = 0; row < nrows; row++) {
                dataType tmp  = alpha * b[row];
                dataType diag = set_fp_value(dataType(0.0), dataType(0.0));
                for (intType i = ia[row] - a_ind; i < ia[row + 1] - a_ind; i++) {
                    intType col = ja[i] - a_ind;
                    if (col < row) {
                        tmp -= a[i] * x_ref[col];
                    }
                    else if (col == row) {
                        diag = a[i];
                    }
                }
                if (diag_val==oneapi::mkl::diag::nonunit)
                    x_ref[row] = tmp / diag;
                else
                    x_ref[row] = tmp;
            }
        }
        else { // uplo::upper
            for (intType row = nrows - 1; row >=0; row--) {
                dataType tmp  = alpha * b[row];
                dataType diag = set_fp_value(dataType(0.0), dataType(0.0));
                for (intType i = ia[row] - a_ind; i < ia[row + 1] - a_ind; i++) {
                    intType col = ja[i] - a_ind;
                    if (col > row) {
                        tmp -= a[i] * x_ref[col];
                    }
                    else if (col == row) {
                        diag = a[i];
                    }
                }
                if (diag_val==oneapi::mkl::diag::nonunit)
                    x_ref[row] = tmp / diag;
                else
                    x_ref[row] = tmp;
            }
        }

        for (intType row = 0; row < nrows; row++) {
            good &= check_result(res[row], x_ref[row], nrows, row);
        }

        std::cout << "\n\t\t sparse::trsv example " << (good ? "passed" : "failed")
                  << "\n\tFinished" << std::endl;

        q.wait_and_throw();

    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;
        good = false;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;
        good = false;
    }

    q.wait();

    // backup cleaning of matrix handle and others for if exceptions happened
    oneapi::mkl::sparse::release_matrix_handle(q, &csrA);

    q.wait();

    return good ? 0 : 1;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# Sparse Triangular Solve Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# x = op(A)^(-1) * b" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# where A is a sparse matrix in CSR format, b and x are "
                 "dense vectors"
              << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::trsv" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "#   std::complex<float>" << std::endl;
    std::cout << "#   std::complex<double>" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type,
//  run_sparse_blas_example is run with all supported data types,
//  if any fail, we move on to the next device.
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {
        try {
            sycl::device my_dev;
            bool my_dev_is_found = false;
            get_sycl_device(my_dev, my_dev_is_found, *it);

            if (my_dev_is_found) {
                std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

                // Catch asynchronous exceptions
                auto exception_handler = [](sycl::exception_list exceptions) {
                    for (std::exception_ptr const &e : exceptions) {
                        try {
                            std::rethrow_exception(e);
                        }
                        catch (sycl::exception const &e) {
                            std::cout << "Caught asynchronous SYCL exception: \n"
                                << e.what() << std::endl;
                        }
                    }
                };

                sycl::queue q(my_dev, exception_handler);

                std::cout << "\tRunning with single precision real data type:" << std::endl;
                status |= run_sparse_blas_example<float, std::int32_t>(q);

                if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                    std::cout << "\tRunning with double precision real data type:" << std::endl;
                    status |= run_sparse_blas_example<double, std::int32_t>(q);
                }

                std::cout << "\tRunning with single precision complex data type:" << std::endl;
                status |= run_sparse_blas_example<std::complex<float>, std::int32_t>(q);

                if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                    std::cout << "\tRunning with double precision complex data type:" << std::endl;
                    status |= run_sparse_blas_example<std::complex<double>, std::int32_t>(q);
                }

            }
            else {
#ifdef FAIL_ON_MISSING_DEVICES
                std::cout << "No " << sycl_device_names[*it]
                    << " devices found; Fail on missing devices "
                    "is enabled.\n";
                return 1;
#else
                std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                    << sycl_device_names[*it] << " tests.\n";
#endif
            }
        }
        catch (sycl::exception const &e) {
            std::cout << "\t\tCaught SYCL exception at driver level: \n" << e.what() << std::endl;
            continue; // stop with device, but move on to other devices
        }
        catch (std::exception const &e) {
            std::cout << "\t\tCaught std exception at driver level: \n" << e.what() << std::endl;
            continue; // stop with device, but move on to other devices
        }


    } // for loop over devices

    mkl_free_buffers();
    return status;
}
