# Copyright (C) 2025 Intel Corporation
# SPDX-License-Identifier: MIT

import os
import platform
from abc import ABC, abstractmethod
from dataclasses import dataclass
from pathlib import Path


@dataclass
class OS:
    LINUX = 'Linux'
    WINDOWS = 'Windows'


class NotAFileError(Exception):
    def __init__(self, msg):
        super().__init__(msg)


class DirectoryIsEmptyError(Exception):
    def __init__(self, msg):
        super().__init__(msg)


class Archive(ABC):

    ZIP = 'zip'
    GZTAR = 'gztar'
    GZ = 'gz'
    TARGZ = 'tar.gz'

    FORMAT = {
        OS.WINDOWS: [ZIP],
        OS.LINUX: [GZTAR]
    }


    def __init__(self, src, dst = None, archive_format = None):
        self._src: str = src
        self._dst: str = dst
        self._format: str = self._get_format(archive_format)
        self._base_name_with_path: str = self._get_archive_base_name_with_path()
        self._base_name: str = self._get_archive_base_name()
        self.__validate()

    @property
    def base_name(self):
        return self._base_name

    @property
    def base_name_with_path(self):
        return self._base_name_with_path

    @property
    def format(self):
        return self._format

    @staticmethod
    def get_archive_suffix(archive_format: str):
        if archive_format in Archive.FORMAT[OS.WINDOWS]:
            return f'.{Archive.ZIP}'
        elif archive_format in Archive.FORMAT[OS.LINUX]:
            return f'.{Archive.TARGZ}'
        else:
            raise ValueError(f'Unsupported archive format: {archive_format}')

    def _get_archive_base_name_with_path(self):
        archive = self._get_archive()
        if not archive:
            return self._src if self._src else self._dst
        if archive.suffix == f'.{Archive.ZIP}':
            archive = archive.with_suffix('')
        elif archive.suffix == f'.{Archive.GZ}':
            archive = archive.with_suffix('').with_suffix('')
        return str(archive)

    def _get_archive_base_name(self):
        return os.path.basename(self._get_archive_base_name_with_path())

    def _get_archive(self):
        archive = None
        if self._is_archive(Path(self._src)):
            archive = Path(self._src)
        elif self._is_archive(Path(self._dst)):
            archive = Path(self._dst)
        return archive

    @staticmethod
    def _is_archive(file_path: Path) -> bool:
        if file_path.suffix == f'.{Archive.ZIP}' or file_path.suffix == f'.{Archive.GZ}':
            return True
        return False

    @staticmethod
    def _get_format(archive_format = None):
        if archive_format and any(archive_format in formats for formats in Archive.FORMAT.values()):
            return archive_format
        if archive_format:
            raise ValueError(f'Unsupported archive format: {archive_format}')
        return Archive.__get_archive_format_for_os()

    @abstractmethod
    def _validate_src(self):
        pass

    @staticmethod
    def __get_archive_format_for_os():
        if platform.system() == OS.WINDOWS:
            return Archive.FORMAT[OS.WINDOWS][0]
        elif platform.system() == OS.LINUX:
            return Archive.FORMAT[OS.LINUX][0]
        else:
            raise ValueError(f'Unsupported OS: {platform.system()}')

    def __validate_archive_format(self):
        if not any(self.format in formats for formats in Archive.FORMAT.values()):
            raise ValueError(f'Unsupported archive extension: {self.format}')

    def __validate(self):
        self._validate_src()
        self.__validate_archive_format()

class ArchiveExtractor(Archive):

    @abstractmethod
    def extract_archive(self):
        pass

    def _validate_src(self):
        if not Path(self._src).exists():
            raise FileNotFoundError(f'Source archive {self._src} does not exist.')
        if not Path(self._src).is_file():
            raise NotAFileError(f'Source {self._src} is not a file.')


class WindowsArchiveExtractor(ArchiveExtractor):

    def extract_archive(self):
        import zipfile
        with zipfile.ZipFile(self._src, 'r') as archive:
            archive.extractall(self._dst)


class LinuxArchiveExtractor(ArchiveExtractor):

    def extract_archive(self):
        import tarfile
        import sys
        with tarfile.open(self._src) as archive:
            if sys.version_info >= (3, 12):
                archive.extractall(self._dst, filter='data')
            else:
                archive.extractall(self._dst)


class ArchiveExtractorFactory:

    archive_extractor = {
        f'.{Archive.ZIP}': WindowsArchiveExtractor,
        f'.{Archive.TARGZ}': LinuxArchiveExtractor,
        f'.{Archive.GZ}': LinuxArchiveExtractor
    }

    @classmethod
    def create(cls, src:str, dst: str = None):
        suffix = cls.get_valid_archive_extensions(src)
        return cls.archive_extractor[suffix](src, dst)

    @classmethod
    def get_valid_archive_extensions(cls, src: str):
        suffixes = Path(src).suffixes
        for i in range(len(suffixes)):
            combined_suffix = ''.join(suffixes[i:])
            if combined_suffix in ArchiveExtractorFactory.archive_extractor.keys():
                return combined_suffix
        raise ValueError(f"No archiver found for {src}")


class ArchiveMaker(Archive):

    def __init__(self, src, dst = None, archive_format = None):
        super().__init__(src, dst, archive_format)

    def make_archive(self):
        import shutil
        shutil.make_archive(self.base_name_with_path, self.format, self._src)

    def _validate_src(self):
        if not Path(self._src).exists():
            raise FileNotFoundError(f'Source file {self._src} does not exist.')
        if not Path(self._src).is_dir():
            raise NotADirectoryError(f'Source {self._src} is not a directory.')
        if not os.listdir(self._src):
            raise DirectoryIsEmptyError(f"Source directory {self._src} is empty.")
