﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/servicecatalog/ServiceCatalogRequest.h>
#include <aws/servicecatalog/ServiceCatalog_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ServiceCatalog {
namespace Model {

/**
 */
class ImportAsProvisionedProductRequest : public ServiceCatalogRequest {
 public:
  AWS_SERVICECATALOG_API ImportAsProvisionedProductRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ImportAsProvisionedProduct"; }

  AWS_SERVICECATALOG_API Aws::String SerializePayload() const override;

  AWS_SERVICECATALOG_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The language code.</p> <ul> <li> <p> <code>jp</code> - Japanese</p> </li>
   * <li> <p> <code>zh</code> - Chinese</p> </li> </ul>
   */
  inline const Aws::String& GetAcceptLanguage() const { return m_acceptLanguage; }
  inline bool AcceptLanguageHasBeenSet() const { return m_acceptLanguageHasBeenSet; }
  template <typename AcceptLanguageT = Aws::String>
  void SetAcceptLanguage(AcceptLanguageT&& value) {
    m_acceptLanguageHasBeenSet = true;
    m_acceptLanguage = std::forward<AcceptLanguageT>(value);
  }
  template <typename AcceptLanguageT = Aws::String>
  ImportAsProvisionedProductRequest& WithAcceptLanguage(AcceptLanguageT&& value) {
    SetAcceptLanguage(std::forward<AcceptLanguageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The product identifier.</p>
   */
  inline const Aws::String& GetProductId() const { return m_productId; }
  inline bool ProductIdHasBeenSet() const { return m_productIdHasBeenSet; }
  template <typename ProductIdT = Aws::String>
  void SetProductId(ProductIdT&& value) {
    m_productIdHasBeenSet = true;
    m_productId = std::forward<ProductIdT>(value);
  }
  template <typename ProductIdT = Aws::String>
  ImportAsProvisionedProductRequest& WithProductId(ProductIdT&& value) {
    SetProductId(std::forward<ProductIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the provisioning artifact.</p>
   */
  inline const Aws::String& GetProvisioningArtifactId() const { return m_provisioningArtifactId; }
  inline bool ProvisioningArtifactIdHasBeenSet() const { return m_provisioningArtifactIdHasBeenSet; }
  template <typename ProvisioningArtifactIdT = Aws::String>
  void SetProvisioningArtifactId(ProvisioningArtifactIdT&& value) {
    m_provisioningArtifactIdHasBeenSet = true;
    m_provisioningArtifactId = std::forward<ProvisioningArtifactIdT>(value);
  }
  template <typename ProvisioningArtifactIdT = Aws::String>
  ImportAsProvisionedProductRequest& WithProvisioningArtifactId(ProvisioningArtifactIdT&& value) {
    SetProvisioningArtifactId(std::forward<ProvisioningArtifactIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The user-friendly name of the provisioned product. The value must be unique
   * for the Amazon Web Services account. The name cannot be updated after the
   * product is provisioned. </p>
   */
  inline const Aws::String& GetProvisionedProductName() const { return m_provisionedProductName; }
  inline bool ProvisionedProductNameHasBeenSet() const { return m_provisionedProductNameHasBeenSet; }
  template <typename ProvisionedProductNameT = Aws::String>
  void SetProvisionedProductName(ProvisionedProductNameT&& value) {
    m_provisionedProductNameHasBeenSet = true;
    m_provisionedProductName = std::forward<ProvisionedProductNameT>(value);
  }
  template <typename ProvisionedProductNameT = Aws::String>
  ImportAsProvisionedProductRequest& WithProvisionedProductName(ProvisionedProductNameT&& value) {
    SetProvisionedProductName(std::forward<ProvisionedProductNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the resource to be imported. It only currently
   * supports CloudFormation stack IDs.</p>
   */
  inline const Aws::String& GetPhysicalId() const { return m_physicalId; }
  inline bool PhysicalIdHasBeenSet() const { return m_physicalIdHasBeenSet; }
  template <typename PhysicalIdT = Aws::String>
  void SetPhysicalId(PhysicalIdT&& value) {
    m_physicalIdHasBeenSet = true;
    m_physicalId = std::forward<PhysicalIdT>(value);
  }
  template <typename PhysicalIdT = Aws::String>
  ImportAsProvisionedProductRequest& WithPhysicalId(PhysicalIdT&& value) {
    SetPhysicalId(std::forward<PhysicalIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier that you provide to ensure idempotency. If multiple
   * requests differ only by the idempotency token, the same response is returned for
   * each repeated request.</p>
   */
  inline const Aws::String& GetIdempotencyToken() const { return m_idempotencyToken; }
  inline bool IdempotencyTokenHasBeenSet() const { return m_idempotencyTokenHasBeenSet; }
  template <typename IdempotencyTokenT = Aws::String>
  void SetIdempotencyToken(IdempotencyTokenT&& value) {
    m_idempotencyTokenHasBeenSet = true;
    m_idempotencyToken = std::forward<IdempotencyTokenT>(value);
  }
  template <typename IdempotencyTokenT = Aws::String>
  ImportAsProvisionedProductRequest& WithIdempotencyToken(IdempotencyTokenT&& value) {
    SetIdempotencyToken(std::forward<IdempotencyTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_acceptLanguage;

  Aws::String m_productId;

  Aws::String m_provisioningArtifactId;

  Aws::String m_provisionedProductName;

  Aws::String m_physicalId;

  Aws::String m_idempotencyToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_acceptLanguageHasBeenSet = false;
  bool m_productIdHasBeenSet = false;
  bool m_provisioningArtifactIdHasBeenSet = false;
  bool m_provisionedProductNameHasBeenSet = false;
  bool m_physicalIdHasBeenSet = false;
  bool m_idempotencyTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace ServiceCatalog
}  // namespace Aws
