"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSvelteSys = createSvelteSys;
const utils_1 = require("./utils");
const fileCollection_1 = require("../../lib/documents/fileCollection");
/**
 * This should only be accessed by TS svelte module resolution.
 */
function createSvelteSys(tsSystem) {
    const fileExistsCache = new fileCollection_1.FileMap();
    function svelteFileExists(path) {
        if ((0, utils_1.isVirtualSvelteFilePath)(path)) {
            const sveltePath = (0, utils_1.toRealSvelteFilePath)(path);
            // First check if there's a `.svelte.d.ts` or `.d.svelte.ts` file, which should take precedence
            const dtsPath = sveltePath.slice(0, -7) + '.svelte.d.ts';
            const dtsPathExists = getOrCreateFileExistCache(dtsPath);
            if (dtsPathExists)
                return false;
            const svelteDtsPathExists = getOrCreateFileExistCache(path);
            if (svelteDtsPathExists)
                return false;
            const sveltePathExists = getOrCreateFileExistCache(sveltePath);
            return sveltePathExists;
        }
        else {
            return false;
        }
    }
    function getOrCreateFileExistCache(path) {
        const cached = fileExistsCache.get(path);
        if (cached !== undefined) {
            return cached;
        }
        const exists = tsSystem.fileExists(path);
        fileExistsCache.set(path, exists);
        return exists;
    }
    function getRealSveltePathIfExists(path) {
        return svelteFileExists(path) ? (0, utils_1.toRealSvelteFilePath)(path) : path;
    }
    const svelteSys = {
        ...tsSystem,
        svelteFileExists,
        getRealSveltePathIfExists,
        fileExists(path) {
            // We need to check if this is a virtual svelte file
            const sveltePathExists = svelteFileExists(path);
            if (sveltePathExists)
                return true;
            return getOrCreateFileExistCache(path);
        },
        readFile(path) {
            // No getSnapshot here, because TS will very rarely call this and only for files that are not in the project
            return tsSystem.readFile(getRealSveltePathIfExists(path));
        },
        readDirectory(path, extensions, exclude, include, depth) {
            const extensionsWithSvelte = extensions ? [...extensions, '.svelte'] : undefined;
            return tsSystem.readDirectory(path, extensionsWithSvelte, exclude, include, depth);
        },
        deleteFile(path) {
            // assumption: never a foo.svelte.ts file next to a foo.svelte file
            fileExistsCache.delete((0, utils_1.ensureRealSvelteFilePath)(path));
            fileExistsCache.delete(path);
            return tsSystem.deleteFile?.(path);
        },
        deleteFromCache(path) {
            // assumption: never a foo.svelte.ts file next to a foo.svelte file
            fileExistsCache.delete((0, utils_1.ensureRealSvelteFilePath)(path));
            fileExistsCache.delete(path);
        }
    };
    if (tsSystem.realpath) {
        const realpath = tsSystem.realpath;
        svelteSys.realpath = function (path) {
            if (svelteFileExists(path)) {
                return realpath((0, utils_1.toRealSvelteFilePath)(path));
            }
            return realpath(path);
        };
    }
    return svelteSys;
}
//# sourceMappingURL=svelte-sys.js.map