"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerADRoutes = registerADRoutes;
var _lodash = require("lodash");
var _constants = require("../utils/constants");
var _helpers = require("../utils/helpers");
var _adHelpers = require("./utils/adHelpers");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * The OpenSearch Contributors require contributions made to
                                                                                                                                                                                                                                                                                                                          * this file be licensed under the Apache-2.0 license or a
                                                                                                                                                                                                                                                                                                                          * compatible open source license.
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * Modifications Copyright OpenSearch Contributors. See
                                                                                                                                                                                                                                                                                                                          * GitHub history for details.
                                                                                                                                                                                                                                                                                                                          */
function registerADRoutes(apiRouter, adService) {
  // create detector
  apiRouter.post('/detectors', adService.putDetector);
  apiRouter.post('/detectors/{dataSourceId}', adService.putDetector);

  // put detector
  apiRouter.put('/detectors/{detectorId}', adService.putDetector);
  apiRouter.put('/detectors/{detectorId}/{dataSourceId}', adService.putDetector);

  // routes not used in the UI, therefore no data source id
  apiRouter.post('/detectors/_search', adService.searchDetector);

  // post search anomaly results
  apiRouter.post('/detectors/results/_search', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{dataSourceId}', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{resultIndex}/{onlyQueryCustomResultIndex}', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{resultIndex}/{onlyQueryCustomResultIndex}/{dataSourceId}', adService.searchResults);

  // list detectors
  apiRouter.get('/detectors/_list', adService.getDetectors);
  apiRouter.get('/detectors/_list/{dataSourceId}', adService.getDetectors);

  // preview detector
  apiRouter.post('/detectors/preview', adService.previewDetector);
  apiRouter.post('/detectors/preview/{dataSourceId}', adService.previewDetector);

  // suggest detector
  apiRouter.post('/detectors/_suggest/{suggestType}', adService.suggestDetector);
  apiRouter.post('/detectors/_suggest/{suggestType}/{dataSourceId}', adService.suggestDetector);

  // get detector anomaly results
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{resultIndex}/{onlyQueryCustomResultIndex}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{resultIndex}/{onlyQueryCustomResultIndex}/{dataSourceId}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{dataSourceId}', adService.getAnomalyResults);

  // delete detector
  apiRouter.delete('/detectors/{detectorId}', adService.deleteDetector);
  apiRouter.delete('/detectors/{detectorId}/{dataSourceId}', adService.deleteDetector);

  // start detector
  apiRouter.post('/detectors/{detectorId}/start', adService.startDetector);
  apiRouter.post('/detectors/{detectorId}/start/{dataSourceId}', adService.startDetector);

  // stop detector
  apiRouter.post('/detectors/{detectorId}/stop/{isHistorical}', adService.stopDetector);
  apiRouter.post('/detectors/{detectorId}/stop/{isHistorical}/{dataSourceId}', adService.stopDetector);
  apiRouter.get('/detectors/{detectorId}/_profile', adService.getDetectorProfile);

  // get detector
  apiRouter.get('/detectors/{detectorId}', adService.getDetector);
  apiRouter.get('/detectors/{detectorId}/{dataSourceId}', adService.getDetector);

  // match detector
  apiRouter.get('/detectors/{detectorName}/_match', adService.matchDetector);
  apiRouter.get('/detectors/{detectorName}/_match/{dataSourceId}', adService.matchDetector);

  // get detector count
  apiRouter.get('/detectors/_count', adService.getDetectorCount);
  apiRouter.get('/detectors/_count/{dataSourceId}', adService.getDetectorCount);

  // post get top anomaly results
  apiRouter.post('/detectors/{detectorId}/_topAnomalies/{isHistorical}', adService.getTopAnomalyResults);
  apiRouter.post('/detectors/{detectorId}/_topAnomalies/{isHistorical}/{dataSourceId}', adService.getTopAnomalyResults);

  // validate detector
  apiRouter.post('/detectors/_validate/{validationType}', adService.validateDetector);
  apiRouter.post('/detectors/_validate/{validationType}/{dataSourceId}', adService.validateDetector);

  // start insights job
  apiRouter.post('/insights/_start', adService.startInsights);
  apiRouter.post('/insights/_start/{dataSourceId}', adService.startInsights);

  // stop insights job
  apiRouter.post('/insights/_stop', adService.stopInsights);
  apiRouter.post('/insights/_stop/{dataSourceId}', adService.stopInsights);

  // get insights status
  apiRouter.get('/insights/_status', adService.getInsightsStatus);
  apiRouter.get('/insights/_status/{dataSourceId}', adService.getInsightsStatus);

  // get insights results
  apiRouter.get('/insights/_results', adService.getInsightsResults);
  apiRouter.get('/insights/_results/{dataSourceId}', adService.getInsightsResults);
}
class AdService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "deleteDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.deleteDetector', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - deleteDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "previewDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.previewDetector', {
          body: requestBody
        });
        const transformedKeys = (0, _helpers.mapKeysDeep)(response, _helpers.toCamel);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            //@ts-ignore
            response: (0, _adHelpers.anomalyResultMapper)(transformedKeys.anomalyResult)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - previewDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "suggestDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          suggestType
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const requestBody = JSON.stringify((0, _adHelpers.convertDetectorKeysToSnakeCase)(request.body));
        const response = await callWithRequest('ad.suggestDetector', {
          body: requestBody,
          suggestType: suggestType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - suggestDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "putDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;

        //@ts-ignore
        const ifSeqNo = request.body.seqNo;
        //@ts-ignore
        const ifPrimaryTerm = request.body.primaryTerm;
        const requestBody = JSON.stringify((0, _adHelpers.convertDetectorKeysToSnakeCase)(request.body));
        let params = {
          detectorId: detectorId,
          ifSeqNo: ifSeqNo,
          ifPrimaryTerm: ifPrimaryTerm,
          body: requestBody
        };
        let response;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        if ((0, _lodash.isNumber)(ifSeqNo) && (0, _lodash.isNumber)(ifPrimaryTerm)) {
          response = await callWithRequest('ad.updateDetector', params);
        } else {
          response = await callWithRequest('ad.createDetector', {
            body: params.body
          });
        }
        const resp = {
          ...response.anomaly_detector,
          id: response._id,
          primaryTerm: response._primary_term,
          seqNo: response._seq_no
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: (0, _adHelpers.convertDetectorKeysToCamelCase)(resp)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - PutDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "validateDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          validationType
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const response = await callWithRequest('ad.validateDetector', {
          body: requestBody,
          validationType: validationType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - validateDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "startInsights", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.startInsights', {});
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response,
            message: 'Insights job started successfully'
          }
        });
      } catch (err) {
        console.log('Anomaly detector - startInsights', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "stopInsights", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.stopInsights', {});
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response,
            message: 'Insights job stopped successfully'
          }
        });
      } catch (err) {
        console.log('Anomaly detector - stopInsights', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getInsightsStatus", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.getInsightsStatus', {});
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getInsightsStatus', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getInsightsResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.getInsightsResults', {});
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getInsightsResults', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const detectorResponse = await callWithRequest('ad.getDetector', {
          detectorId
        });

        // Populating static detector fields
        const staticFields = {
          id: detectorResponse._id,
          primaryTerm: detectorResponse._primary_term,
          seqNo: detectorResponse._seq_no,
          ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse.anomaly_detector)
        };

        // Get real-time and historical task info to populate the
        // task and job-related fields
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};
        try {
          const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
          realtimeTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, true)
          });
          historicalTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTask = (0, _lodash.get)((0, _lodash.get)(realtimeTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const historicalTask = (0, _lodash.get)((0, _lodash.get)(historicalTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const taskAndJobFields = (0, _adHelpers.convertTaskAndJobFieldsToCamelCase)(realtimeTask, historicalTask, detectorResponse.anomaly_detector_job);

        // Combine the static and task-and-job-related fields into
        // a final response
        const finalResponse = {
          ...staticFields,
          ...taskAndJobFields
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: finalResponse
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get detector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "startDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        var _request$body, _request$body2;
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        //@ts-ignore
        const startTime = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.startTime;
        //@ts-ignore
        const endTime = (_request$body2 = request.body) === null || _request$body2 === void 0 ? void 0 : _request$body2.endTime;
        let requestParams = {
          detectorId: detectorId
        };
        let requestPath = 'ad.startDetector';
        // If a start and end time are passed: we want to start a historical analysis
        if ((0, _lodash.isNumber)(startTime) && (0, _lodash.isNumber)(endTime)) {
          requestParams = {
            ...requestParams,
            body: {
              start_time: startTime,
              end_time: endTime
            }
          };
          requestPath = 'ad.startHistoricalDetector';
        }
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, requestParams);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - startDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "stopDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.stopHistoricalDetector' : 'ad.stopDetector';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - stopDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectorProfile", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.detectorProfile', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - detectorProfile', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify(request.body);
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectors = (0, _lodash.get)(response, 'hits.hits', []).map(detector => ({
          ...(0, _adHelpers.convertDetectorKeysToCamelCase)(detector._source),
          id: detector._id,
          seqNo: detector._seq_no,
          primaryTerm: detector._primary_term
        }));
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectors
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        var {
          resultIndex,
          onlyQueryCustomResultIndex
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
          // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
          resultIndex = '';
        }
        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const requestBody = JSON.stringify(request.body);
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = !resultIndex ? await callWithRequest('ad.searchResults', {
          body: requestBody
        }) : await callWithRequest('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search anomaly result', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectors", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          from = 0,
          size = 20,
          search = '',
          indices = '',
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = 'name',
          dataSourceId = ''
        } = request.query;
        const mustQueries = [];
        if (search.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['name', 'description'],
              default_operator: 'AND',
              query: `*${search.trim().split('-').join('* *')}*`
            }
          });
        }
        if (indices.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['indices'],
              default_operator: 'AND',
              query: `*${indices.trim().split('-').join('* *')}*`
            }
          });
        }
        //Allowed sorting columns
        const sortQueryMap = {
          name: {
            'name.keyword': sortDirection
          },
          indices: {
            'indices.keyword': sortDirection
          },
          lastUpdateTime: {
            last_update_time: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }
        //Preparing search request
        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              must: mustQueries
            }
          }
        };
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0);

        //Get all detectors from search detector API
        const allDetectors = (0, _lodash.get)(response, 'hits.hits', []).reduce((acc, detectorResponse) => ({
          ...acc,
          [detectorResponse._id]: {
            id: detectorResponse._id,
            primaryTerm: detectorResponse._primary_term,
            seqNo: detectorResponse._seq_no,
            ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse._source)
          }
        }), {});
        const allDetectorsMap = Object.values(allDetectors).reduce((acc, detector) => ({
          ...acc,
          [detector.id]: detector
        }), {});

        //Given each detector from previous result, get aggregation to power list
        const allDetectorIds = Object.keys(allDetectorsMap);
        let requestParams = {
          // If specifying result index, will query anomaly result from both default and custom result indices.
          // If no valid result index specified, just query anomaly result from default result index.
          // Here we specify custom AD result index prefix pattern to query all custom result indices.
          resultIndex: _constants.CUSTOM_AD_RESULT_INDEX_PREFIX + '*',
          onlyQueryCustomResultIndex: 'false'
        };
        const aggregationResult = await callWithRequest('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: (0, _adHelpers.getResultAggregationQuery)(allDetectorIds, {
            from,
            size,
            sortField,
            sortDirection,
            search,
            indices
          })
        });
        const aggsDetectors = (0, _lodash.get)(aggregationResult, 'aggregations.unique_detectors.buckets', []).reduce((acc, agg) => {
          return {
            ...acc,
            [agg.key]: {
              ...allDetectorsMap[agg.key],
              totalAnomalies: agg.total_anomalies_in_24hr.doc_count,
              lastActiveAnomaly: agg.latest_anomaly_time.value
            }
          };
        }, {});

        // Aggregation will not return values where anomalies for detectors are not generated, loop through it and fill values with 0
        const unUsedDetectors = (0, _lodash.pullAll)(allDetectorIds, Object.keys(aggsDetectors)).reduce((acc, unusedDetector) => {
          return {
            ...acc,
            [unusedDetector]: {
              ...allDetectorsMap[unusedDetector],
              totalAnomalies: 0,
              lastActiveAnomaly: 0
            }
          };
        }, {});

        // If sorting criteria is from the aggregation manage pagination in memory.
        let finalDetectors = (0, _lodash.orderBy)({
          ...aggsDetectors,
          ...unUsedDetectors
        }, [sortField], [sortDirection]);
        if (!sortQueryMap[sortField]) {
          finalDetectors = Object.values(finalDetectors).slice(from, from + size).reduce((acc, detector) => ({
            ...acc,
            [detector.id]: detector
          }), {});
        }

        // Fetch the latest realtime and historical tasks for all detectors
        // using terms aggregations
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};
        try {
          realtimeTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(true)
          });
          historicalTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTasks = (0, _lodash.get)(realtimeTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              realtimeTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});
        const historicalTasks = (0, _lodash.get)(historicalTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              historicalTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});

        // Get real-time and historical task info by looping through each detector & retrieving
        //    - curState by getting real-time task state
        //    - enabledTime by getting real-time task's execution_start time
        //    - taskId by getting historical task's _id
        finalDetectors.forEach(detector => {
          const realtimeTask = (0, _lodash.get)(realtimeTasks[detector.id], 'realtimeTask._source');
          detector.curState = (0, _adHelpers.getTaskState)(realtimeTask);
          detector.enabledTime = (0, _lodash.get)(realtimeTask, 'execution_start_time');
          detector.taskId = (0, _lodash.get)(historicalTasks[detector.id], 'historicalTask._id');
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectorList: Object.values(finalDetectors)
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectorList: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      let {
        id,
        isHistorical,
        resultIndex,
        onlyQueryCustomResultIndex
      } = request.params;
      const {
        dataSourceId = ''
      } = request.params;
      if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
        // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
        resultIndex = '';
      }
      isHistorical = JSON.parse(isHistorical);

      // Search by task id if historical, or by detector id if realtime
      const searchTerm = isHistorical ? {
        task_id: id
      } : {
        detector_id: id
      };
      try {
        const {
          from = 0,
          size = 20,
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = _constants.AD_DOC_FIELDS.DATA_START_TIME,
          startTime = 0,
          endTime = 0,
          fieldName = '',
          anomalyThreshold = -1,
          entityList = ''
        } = request.query;
        const entityListAsObj = entityList.length === 0 ? {} : JSON.parse(entityList);
        const entityFilters = (0, _lodash.isEmpty)(entityListAsObj) ? [] : (0, _adHelpers.getFiltersFromEntityList)(entityListAsObj);

        //Allowed sorting columns
        const sortQueryMap = {
          anomalyGrade: {
            anomaly_grade: sortDirection
          },
          confidence: {
            confidence: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_START_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_START_TIME]: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_END_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_END_TIME]: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }

        //Preparing search request
        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              filter: [{
                term: searchTerm
              }, {
                range: {
                  anomaly_grade: {
                    gt: anomalyThreshold
                  }
                }
              }, ...entityFilters]
            }
          }
        };

        // If querying RT results: remove any results that include a task_id, as this indicates
        // a historical result from a historical task.
        if (!isHistorical) {
          requestBody.query.bool = {
            ...requestBody.query.bool,
            ...{
              must_not: {
                exists: {
                  field: 'task_id'
                }
              }
            }
          };
        }
        try {
          const filterSize = requestBody.query.bool.filter.length;
          if (fieldName) {
            (startTime || endTime) && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.format`, 'epoch_millis');
            startTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.gte`, startTime);
            endTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.lte`, endTime);
          }
        } catch (error) {
          console.log('wrong date range filter', error);
        }
        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = !resultIndex ? await callWithRequest('ad.searchResults', {
          body: requestBody
        }) : await callWithRequest('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        const totalResults = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectorResult = [];
        const featureResult = {};
        (0, _lodash.get)(response, 'hits.hits', []).forEach(result => {
          detectorResult.push({
            startTime: result._source.data_start_time,
            endTime: result._source.data_end_time,
            plotTime: result._source.data_end_time,
            contributions: result._source.anomaly_grade > 0 ? result._source.relevant_attribution : {},
            confidence: result._source.confidence != null && result._source.confidence !== 'NaN' && result._source.confidence > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.confidence)) : 0,
            anomalyGrade: result._source.anomaly_grade != null && result._source.anomaly_grade !== 'NaN' && result._source.anomaly_grade > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.anomaly_grade)) : 0,
            ...(result._source.entity != null ? {
              entity: result._source.entity
            } : {}),
            // TODO: we should refactor other places to read feature data from
            // AnomalyResult, instead of having separate FeatureData which is hard
            // to know feature data belongs to which anomaly result
            features: this.getFeatureData(result)
          });
          result._source.feature_data.forEach(featureData => {
            if (!featureResult[featureData.feature_id]) {
              featureResult[featureData.feature_id] = [];
            }
            featureResult[featureData.feature_id].push({
              startTime: result._source.data_start_time,
              endTime: result._source.data_end_time,
              plotTime: result._source.data_end_time,
              data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0,
              name: featureData.feature_name,
              expectedValue: this.getExpectedValue(result, featureData)
            });
          });
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalAnomalies: totalResults,
              results: detectorResult,
              featureResults: featureResult
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get results', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getTopAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.topHistoricalAnomalyResults' : 'ad.topAnomalyResults';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, {
          detectorId: detectorId,
          body: request.body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getTopAnomalyResults', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "matchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorName
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.matchDetector', {
          detectorName
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - matchDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectorCount", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.detectorCount');
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getDetectorCount', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getFeatureData", rawResult => {
      const featureResult = {};
      rawResult._source.feature_data.forEach(featureData => {
        featureResult[featureData.feature_id] = {
          startTime: rawResult._source.data_start_time,
          endTime: rawResult._source.data_end_time,
          plotTime: rawResult._source.data_end_time,
          data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0,
          name: featureData.feature_name,
          expectedValue: this.getExpectedValue(rawResult, featureData)
        };
      });
      return featureResult;
    });
    _defineProperty(this, "getExpectedValue", (rawResult, featureData) => {
      let expectedValue = featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0;
      if (rawResult._source.anomaly_grade > 0) {
        const expectedValueList = rawResult._source.expected_values;
        if ((expectedValueList === null || expectedValueList === void 0 ? void 0 : expectedValueList.length) > 0) {
          expectedValueList[0].value_list.forEach(expect => {
            if (expect.feature_id === featureData.feature_id) {
              expectedValue = expect.data;
            }
          });
        }
      }
      return expectedValue;
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.default = AdService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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