/*******************************************************************************
* Copyright 2019-2023 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP API oneapi::mkl::sparse::symv to perform symmetrical
*       sparse matrix-vector  multiplication on a SYCL device (CPU, GPU).
*       This example uses a sparse matrix in CSR format.
*
*       y = alpha * A * x + beta * y
*
*       The supported floating point data types for symv are:
*           float
*           double
*           std::complex<float>
*           std::complex<double>
*
*       The supported matrix formats for symv are:
*           CSR
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

// local includes
#include "common_for_examples.hpp"
#include "./include/common_for_sparse_examples.hpp"

//
// example for Symmetric Sparse Matrix-Vector Multiply consisting of
// initialization of A matrix, x and y vectors as well as
// scalars alpha and beta.  Then the product
//
// y = alpha * A * x + beta * y
//
// is performed and finally the results are post processed.
//

template <typename fp, typename intType>
int run_sparse_matrix_vector_multiply_example(const sycl::device &dev)
{
    // Initialize data for Sparse Matrix-Vector Multiply
    oneapi::mkl::uplo uplo_val             = oneapi::mkl::uplo::lower;
    oneapi::mkl::index_base index_base_val = oneapi::mkl::index_base::zero;

    intType int_index = (index_base_val == oneapi::mkl::index_base::zero ? 0 : 1);

    // Matrix data size
    intType size  = 4;
    intType nrows = size * size * size;

    // Input matrix in CSR format
    std::vector<intType, mkl_allocator<intType, 64>> ia;
    std::vector<intType, mkl_allocator<intType, 64>> ja;
    std::vector<fp, mkl_allocator<fp, 64>> a;

    ia.resize(nrows + 1);
    ja.resize(27 * nrows);
    a.resize(27 * nrows);

    generate_sparse_matrix<fp, intType>(size, ia, ja, a, int_index);
    intType nnz = ia[nrows] - int_index;

    // Vectors x and y
    std::vector<fp, mkl_allocator<fp, 64>> x;
    std::vector<fp, mkl_allocator<fp, 64>> y;
    std::vector<fp, mkl_allocator<fp, 64>> z;
    x.resize(nrows);
    y.resize(nrows);
    z.resize(nrows);

    // Init vectors x and y
    for (int i = 0; i < nrows; i++) {
        x[i] = set_fp_value(fp(1.0), fp(0.0));
        y[i] = set_fp_value(fp(0.0), fp(0.0));
        z[i] = set_fp_value(fp(0.0), fp(0.0));
    }

    // Set scalar fp values
    fp alpha = set_fp_value(fp(1.0), fp(0.0));
    fp beta  = set_fp_value(fp(0.0), fp(0.0));

    // Catch asynchronous exceptions
    auto exception_handler = [](sycl::exception_list exceptions) {
        for (std::exception_ptr const &e : exceptions) {
            try {
                std::rethrow_exception(e);
            }
            catch (sycl::exception const &e) {
                std::cout << "Caught asynchronous SYCL "
                             "exception during sparse::symv:\n"
                          << e.what() << std::endl;
            }
        }
    };

    //
    // Execute Matrix Multiply
    //

    std::cout << "\n\t\tsparse::symv parameters:\n";
    std::cout << "\t\t\tuplo_val = " << uplo_val << std::endl;
    std::cout << "\t\t\tindexing = " << index_base_val << std::endl;
    std::cout << "\t\t\tnrows    = " << nrows << std::endl;
    std::cout << "\t\t\talpha    = " << alpha << ", beta = " << beta << std::endl;


    // create execution queue and buffers of matrix data
    sycl::queue main_queue(dev, exception_handler);

    sycl::buffer<intType, 1> ia_buffer(ia.data(), (nrows + 1));
    sycl::buffer<intType, 1> ja_buffer(ja.data(), nnz);
    sycl::buffer<fp, 1> a_buffer(a.data(), nnz);
    sycl::buffer<fp, 1> x_buffer(x.data(), x.size());
    sycl::buffer<fp, 1> y_buffer(y.data(), y.size());

    // create and initialize handle for a Sparse Matrix in CSR format
    oneapi::mkl::sparse::matrix_handle_t handle = nullptr;

    try {
        oneapi::mkl::sparse::init_matrix_handle(&handle);

        oneapi::mkl::sparse::set_csr_data(main_queue, handle, nrows, nrows, index_base_val,
                                          ia_buffer, ja_buffer, a_buffer);

        // add oneapi::mkl::sparse::symv to execution queue
        oneapi::mkl::sparse::symv(main_queue, oneapi::mkl::uplo::lower, alpha, handle, x_buffer,
                                  beta, y_buffer);

        oneapi::mkl::sparse::release_matrix_handle(main_queue, &handle);
    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &handle);
        return 1;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &handle);
        return 1;
    }

    main_queue.wait();

    //
    // Post Processing
    //

    auto res = y_buffer.get_host_access(sycl::read_only);
    for (intType row = 0; row < nrows; row++) {
        fp tmp = set_fp_value(fp(0.0), fp(0.0));
        for (intType i = ia[row] - int_index; i < ia[row + 1] - int_index; i++) {
            tmp += a[i] * x[ja[i] - int_index];
        }
        z[row] = alpha * tmp + beta * z[row];
    }

    bool good = true;
    for (intType row = 0; row < nrows; row++) {
        good &= check_result(res[row], z[row], nrows, row);
    }

    std::cout << "\n\t\t sparse::symv example " << (good ? "passed" : "failed")
              << "\n\tFinished" << std::endl;

    if (!good) return 1;

    return 0;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "#Symmetrical Sparse Matrix-Vector Multiply Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# y = alpha * A * x + beta * y" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# where A is a sparse matrix in CSR format, x and y are "
                 "dense vectors"
              << std::endl;
    std::cout << "# and alpha, beta are floating point type precision scalars." << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::symv" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "#   std::complex<float>" << std::endl;
    std::cout << "#   std::complex<double>" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type, MatrixMultiplyExample
//  runs is with all supported data types
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {

        sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, *it);

        if (my_dev_is_found) {
            std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

            std::cout << "\tRunning with single precision real data type:" << std::endl;
            status = run_sparse_matrix_vector_multiply_example<float, std::int32_t>(my_dev);
            if(status != 0) return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision real data type:" << std::endl;
                status = run_sparse_matrix_vector_multiply_example<double, std::int32_t>(my_dev);
                if(status != 0) return status;
            }

            std::cout << "\tRunning with single precision complex data type:" << std::endl;
            status = run_sparse_matrix_vector_multiply_example<std::complex<float>, std::int32_t>(my_dev);
            if(status != 0) return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision complex data type:" << std::endl;
                status = run_sparse_matrix_vector_multiply_example<std::complex<double>, std::int32_t>(my_dev);
                if(status != 0) return status;
            }
        }
        else {
#ifdef FAIL_ON_MISSING_DEVICES
            std::cout << "No " << sycl_device_names[*it]
                      << " devices found; Fail on missing devices "
                         "is enabled.\n";
            return 1;
#else
            std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                      << sycl_device_names[*it] << " tests.\n";
#endif
        }
    }

    mkl_free_buffers();
    return status;
}
