/*******************************************************************************
* Copyright 2021-2023 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP USM API oneapi::mkl::sparse::sort_matrix to perform general
*       sparse matrix sorting on a SYCL device (CPU, GPU). This example uses a
*       sparse matrix in CSR format.
*
*       The input matrix is sorted in-place and no temporary workspace is used.
*
*       The supported floating point data types for sort_matrix matrix data are:
*           float
*           double
*           std::complex<float>
*           std::complex<double>
*
*       The supported matrix formats for sort_matrix are:
*           CSR
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

// local includes
#include "common_for_examples.hpp"
#include "./include/common_for_sparse_examples.hpp"

//
// Main example for Sparse Matrix sorting
//
template <typename fp, typename intType>
int run_sparse_matrix_sorting_example(const sycl::device &dev)
{
    // Catch asynchronous exceptions
    auto exception_handler = [](sycl::exception_list exceptions) {
        for (std::exception_ptr const &e : exceptions) {
            try {
                std::rethrow_exception(e);
            }
            catch (sycl::exception const &e) {
                std::cout << "Caught asynchronous SYCL "
                             "exception during sparse::gemv:\n"
                          << e.what() << std::endl;
            }
        }
    };

    // create execution main_queue
    sycl::queue main_queue(dev, exception_handler);
    sycl::context cxt = main_queue.get_context();

    // Initialize data for Sparse Matrix
    oneapi::mkl::index_base a_index = oneapi::mkl::index_base::one;
    intType size = 4;
    intType a_nrows = size * size * size;
    intType a_ncols = a_nrows;
    intType a_nnz   = 27 * a_nrows;

    //
    // array memory management tools
    //
    std::vector<intType *> int_ptr_vec;
    std::vector<fp *> fp_ptr_vec;
    std::vector<std::int64_t *> i64_ptr_vec;
    std::vector<void *> void_ptr_vec;

    //
    // setup A data locally in CSR format
    //
    std::vector<intType, mkl_allocator<intType, 64>> ia;
    std::vector<intType, mkl_allocator<intType, 64>> ja;
    std::vector<fp, mkl_allocator<fp, 64>> a;

    ia.resize(a_nrows + 1);
    ja.resize(27 * a_nrows);
    a.resize(27 * a_nrows);

    intType a_ind = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
    generate_sparse_matrix<fp, intType>(size, ia, ja, a, a_ind);
    a_nnz = ia[a_nrows] - a_ind;
    shuffle_matrix_data<fp, intType>(ia, ja, a, a_ind, a_nrows, a_nnz);

    intType *a_rowptr = (intType *)malloc_shared((a_nrows + 1) * sizeof(intType), dev, cxt);
    intType *a_colind = (intType *)malloc_shared((a_nnz) * sizeof(intType), dev, cxt);
    fp *a_val         = (fp *)malloc_shared((a_nnz) * sizeof(fp), dev, cxt);

    if (!a_rowptr || !a_colind || !a_val)
        throw std::runtime_error("Failed to allocate USM memory");

    int_ptr_vec.push_back(a_rowptr);
    int_ptr_vec.push_back(a_colind);
    fp_ptr_vec.push_back(a_val);

    // copy data to USM arrays
    for (intType i = 0; i < a_nrows + 1; ++i) {
        a_rowptr[i] = ia[i];
    }
    for (intType i = 0; i < a_nnz; ++i) {
        a_colind[i] = ja[i];
        a_val[i]    = a[i];
    }

    //
    // other workspaces and arrays for sort_matrix process
    //
    std::int64_t *sizeTempBuffer = nullptr;
    void *tempBuffer = nullptr;


    //
    // Execute Matrix Sort
    //

    const double ave_nnz_per_row = static_cast<double>(a_nnz)/a_nrows;

    intType max_nnz_per_row = 0;
    intType min_nnz_per_row = a_nnz;
    for (intType row = 0; row < a_nrows; ++row) {
        const intType loc_nnz_per_row = a_rowptr[row+1] - a_rowptr[row];

        if (loc_nnz_per_row > max_nnz_per_row) 
            max_nnz_per_row = loc_nnz_per_row;

        if (loc_nnz_per_row < min_nnz_per_row) 
            min_nnz_per_row = loc_nnz_per_row;
    }

    std::cout << "\n\t\tsparse::sort_matrix parameters:\n";
    std::cout << "\t\t\tA_nrows = A_ncols = " << a_nrows << std::endl;
    std::cout << "\t\t\tA_index = " << a_index << std::endl;
    std::cout << "\t\t\tA_nnz   = " << a_nnz << std::endl;
    std::cout << "\t\t\t\tmin_nnz_per_row = " << min_nnz_per_row << std::endl;
    std::cout << "\t\t\t\tave_nnz_per_row = " << ave_nnz_per_row << std::endl;
    std::cout << "\t\t\t\tmax_nnz_per_row = " << max_nnz_per_row << std::endl;


    oneapi::mkl::sparse::matrix_handle_t A = nullptr;

    // print out first bit of C solution
    sycl::event ev_input = main_queue.submit([&](sycl::handler &cgh) {
        auto kernel = [=]() {
            std::cout << "[INPUT] Unsorted A matrix [first two rows]:" << std::endl;
            intType a_ind = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
            for (intType row = 0; row < std::min(static_cast<intType>(2), a_nrows); ++row) {
                for (intType j = a_rowptr[row] - a_ind; j < a_rowptr[row + 1] - a_ind; ++j) {
                    intType col = a_colind[j];
                    fp val  = a_val[j];
                    std::cout << "C(" << row + a_ind << ", " << col << ") = " << val
                              << std::endl;
                }
            }
        };
        cgh.host_task(kernel);
    });

    try {
        oneapi::mkl::sparse::init_matrix_handle(&A);

        auto ev_set = oneapi::mkl::sparse::set_csr_data(main_queue, A, a_nrows, a_ncols, a_index, a_rowptr, a_colind, a_val, {});

        // Sort the sparse matrix
        auto ev_sort = oneapi::mkl::sparse::sort_matrix(main_queue, A, {ev_set, ev_input});

        // cleanup the matrix_handle, as we only need the arrays for future steps
        auto ev_release = oneapi::mkl::sparse::release_matrix_handle(main_queue, &A, {ev_sort});

        // print out first bit of output sorted matrix
        sycl::event ev_print = main_queue.submit([&](sycl::handler &cgh) {
            cgh.depends_on({ev_sort});
            auto kernel = [=]() {
                std::cout << "[OUTPUT] Sorted A matrix [first two rows]:" << std::endl;
                intType a_ind = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
                for (intType row = 0; row < std::min(static_cast<intType>(2), a_nrows); ++row) {
                    for (intType j = a_rowptr[row] - a_ind; j < a_rowptr[row + 1] - a_ind; ++j) {
                        intType col = a_colind[j];
                        fp val = a_val[j];
                        std::cout << "C(" << row + a_ind << ", " << col << ") = " << val
                                  << std::endl;
                    }
                }
            };
            cgh.host_task(kernel);
        });

        ev_release.wait();
        ev_print.wait();
    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A).wait();
        cleanup_arrays<fp, intType>(fp_ptr_vec, int_ptr_vec, cxt);

        return 1;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A).wait();
        cleanup_arrays<fp, intType>(fp_ptr_vec, int_ptr_vec, cxt);

        return 1;
    }

    //
    // Post Processing
    //

    cleanup_arrays<fp, intType>(fp_ptr_vec, int_ptr_vec, i64_ptr_vec, void_ptr_vec, cxt);

    return 0;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# CSR Matrix Sorting Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::sort_matrix" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#   sparse::init_matrix_handle" << std::endl;
    std::cout << "#   sparse::set_csr_data" << std::endl;
    std::cout << "#   sparse::release_matrix_handle" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "#   std::complex<float>" << std::endl;
    std::cout << "#   std::complex<double>" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type, MatrixMultiplyExample
//  runs is with all supported data types
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {

        sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, *it);

        if (my_dev_is_found) {
            std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

            std::cout << "\tRunning with single precision real data type:" << std::endl;
            status = run_sparse_matrix_sorting_example<float, std::int32_t>(my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision real data type:" << std::endl;
                status = run_sparse_matrix_sorting_example<double, std::int32_t>(
                        my_dev);
                if (status != 0)
                    return status;
            }

            std::cout << "\tRunning with single precision complex data type:" << std::endl;
            status = run_sparse_matrix_sorting_example<std::complex<float>, std::int32_t>(my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision complex data type:" << std::endl;
                status = run_sparse_matrix_sorting_example<std::complex<double>, std::int32_t>(
                        my_dev);
                if (status != 0)
                    return status;
            }
        }
        else {
#ifdef FAIL_ON_MISSING_DEVICES
            std::cout << "No " << sycl_device_names[*it]
                      << " devices found; Fail on missing devices "
                         "is enabled.\n";
            return 1;
#else
            std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                      << sycl_device_names[*it] << " tests.\n";
#endif
        }
    }

    return status;
}
