/*******************************************************************************
* Copyright 2022-2023 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP USM API, oneapi::mkl::sparse::omatcopy, to perform an out-of-place
*       copy/transpose operation on a sparse matrix and on a SYCL device (CPU,
*       GPU). This example uses sparse matrices in CSR format.
*
*           B = op(A)
*
*       where op() is defined by one of
*           oneapi::mkl::transpose::{nontrans,trans,conjtrans}
*
*       The supported floating point data types for omatcopy matrix data are:
*           float
*           double
*           std::complex<float>
*           std::complex<double>
*
*       The supported matrix formats for omatcopy are:
*           CSR
*           COO
*
*       Note that omatcopy does not support matrix format conversion (e.g. CSR
*       to COO), and both A and B must be in the same matrix format.
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

// local includes
#include "common_for_examples.hpp"
#include "./include/common_for_sparse_examples.hpp"

//
// Main example for Sparse Matrix copy/transpose: B = op(A)
//
template <typename fpType, typename intType>
int run_sparse_matrix_omatcopy_example(const sycl::device &dev)
{
    // Catch asynchronous exceptions
    auto exception_handler = [](sycl::exception_list exceptions) {
        for (std::exception_ptr const &e : exceptions) {
            try {
                std::rethrow_exception(e);
            }
            catch (sycl::exception const &e) {
                std::cout << "Caught asynchronous SYCL "
                             "exception during sparse::gemv:\n"
                          << e.what() << std::endl;
            }
        }
    };

    // create execution main_queue
    sycl::queue main_queue(dev, exception_handler);
    sycl::context cxt = main_queue.get_context();

    // Initialize data for Sparse Matrix
    oneapi::mkl::transpose opA;
    if constexpr (is_complex<fpType>()) {
        opA = oneapi::mkl::transpose::conjtrans;
    }
    else {
        opA = oneapi::mkl::transpose::trans;
    }
    oneapi::mkl::index_base a_index = oneapi::mkl::index_base::zero;
    oneapi::mkl::index_base b_index = oneapi::mkl::index_base::one;
    intType a_ind                   = a_index == oneapi::mkl::index_base::zero ? 0 : 1;
    intType b_ind                   = b_index == oneapi::mkl::index_base::zero ? 0 : 1;

    //
    // set up dimensions of matrix products
    //
    intType size = 4;

    intType a_nrows = size * size * size;
    intType a_ncols = a_nrows;
    intType a_nnz   = 27 * a_nrows;
    intType b_nrows = (opA == oneapi::mkl::transpose::nontrans ? a_nrows : a_ncols);
    intType b_ncols = (opA == oneapi::mkl::transpose::nontrans ? a_ncols : a_nrows);
    intType b_nnz   = a_nnz;

    //
    // array memory management tools
    //
    std::vector<intType *> int_ptr_vec;
    std::vector<fpType *> fp_ptr_vec;
    std::vector<std::int64_t *> i64_ptr_vec;
    std::vector<void *> void_ptr_vec;

    //
    // setup A data locally in CSR format
    //
    std::vector<intType, mkl_allocator<intType, 64>> ia;
    std::vector<intType, mkl_allocator<intType, 64>> ja;
    std::vector<fpType, mkl_allocator<fpType, 64>> a;

    ia.resize(a_nrows + 1);
    ja.resize(27 * a_nrows);
    a.resize(27 * a_nrows);

    generate_sparse_matrix<fpType, intType>(size, ia, ja, a, a_ind);
    a_nnz = ia[a_nrows] - a_ind;

    // USM arrays for A matrix
    intType *a_rowptr, *a_colind;
    fpType *a_val;

    a_rowptr = (intType *)malloc_shared((a_nrows + 1) * sizeof(intType), dev, cxt);
    a_colind = (intType *)malloc_shared((a_nnz) * sizeof(intType), dev, cxt);
    a_val    = (fpType *)malloc_shared((a_nnz) * sizeof(fpType), dev, cxt);

    if (!a_rowptr || !a_colind || !a_val)
        throw std::runtime_error("Failed to allocate USM memory");

    int_ptr_vec.push_back(a_rowptr);
    int_ptr_vec.push_back(a_colind);
    fp_ptr_vec.push_back(a_val);

    // copy data to USM arrays
    for (intType i = 0; i < a_nrows + 1; ++i) {
        a_rowptr[i] = ia[i];
    }
    for (intType i = 0; i < a_nnz; ++i) {
        a_colind[i] = ja[i];
        a_val[i]    = a[i];
    }

    //
    // setup B data locally in CSR format
    // USM arrays for B (=op(A)) matrix
    //
    intType *b_rowptr, *b_colind;
    fpType *b_val;

    b_rowptr = (intType *)malloc_shared((b_nrows + 1) * sizeof(intType), dev, cxt);
    b_colind = (intType *)malloc_shared((b_nnz) * sizeof(intType), dev, cxt);
    b_val    = (fpType *)malloc_shared((b_nnz) * sizeof(fpType), dev, cxt);

    if (!b_rowptr || !b_colind || !b_val)
        throw std::runtime_error("Failed to allocate USM memory");

    int_ptr_vec.push_back(b_rowptr);
    int_ptr_vec.push_back(b_colind);
    fp_ptr_vec.push_back(b_val);

    //
    // Execute Matrix Copy/Transpose
    //

    std::cout << "\n\t\tsparse::omatcopy parameters:\n";
    std::cout << "\t\t\topA = " << opA << std::endl;

    std::cout << "\t\t\tA_nrows = A_ncols = " << a_nrows << std::endl;
    std::cout << "\t\t\tB_nrows = B_ncols = " << b_nrows << std::endl;

    std::cout << "\t\t\tA_index = " << a_index << std::endl;
    std::cout << "\t\t\tB_index = " << b_index << std::endl;

    oneapi::mkl::sparse::matrix_handle_t A = nullptr;
    oneapi::mkl::sparse::matrix_handle_t B = nullptr;

    try {
        oneapi::mkl::sparse::init_matrix_handle(&A);
        oneapi::mkl::sparse::init_matrix_handle(&B);

        auto ev_setA = oneapi::mkl::sparse::set_csr_data(main_queue, A, a_nrows, a_ncols, a_index, a_rowptr, a_colind, a_val);
        auto ev_setB = oneapi::mkl::sparse::set_csr_data(main_queue, B, b_nrows, b_ncols, b_index, b_rowptr, b_colind, b_val);

        // B = op(A)
        auto ev_trans = oneapi::mkl::sparse::omatcopy(
                main_queue, opA, A, B, {ev_setA, ev_setB});

        // Print portion of B solution
        sycl::event ev_print = main_queue.submit([&](sycl::handler &cgh) {
            cgh.depends_on({ev_trans});
            auto kernel = [=]() {
                std::cout << "B matrix [first two rows]:" << std::endl;
                for (intType row = 0; row < std::min(static_cast<intType>(2), b_nrows); ++row) {
                    for (intType j = b_rowptr[row] - b_ind; j < b_rowptr[row + 1] - b_ind; ++j) {
                        intType col = b_colind[j];
                        fpType val  = b_val[j];
                        std::cout << "B(" << row + b_ind << ", " << col << ") = " << val
                                  << std::endl;
                    }
                }
            };
            cgh.host_task(kernel);
        });

        // clean up
        auto ev_release_A = oneapi::mkl::sparse::release_matrix_handle(main_queue, &A, {ev_print});
        auto ev_release_B = oneapi::mkl::sparse::release_matrix_handle(main_queue, &B, {ev_print});

        ev_release_A.wait();
        ev_release_B.wait();
    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A).wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B).wait();

        cleanup_arrays<fpType, intType>(fp_ptr_vec, int_ptr_vec, i64_ptr_vec, void_ptr_vec, cxt);

        return 1;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;

        main_queue.wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &A).wait();
        oneapi::mkl::sparse::release_matrix_handle(main_queue, &B).wait();

        cleanup_arrays<fpType, intType>(fp_ptr_vec, int_ptr_vec, i64_ptr_vec, void_ptr_vec, cxt);

        return 1;
    }

    cleanup_arrays<fpType, intType>(fp_ptr_vec, int_ptr_vec, i64_ptr_vec, void_ptr_vec, cxt);

    return 0;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# Sparse Matrix Copy/Transpose/Conjugate-transpose Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#    B = op(A)" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# where A and B are sparse matrices in CSR format" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::omatcopy" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "#   sparse::init_matrix_handle" << std::endl;
    std::cout << "#   sparse::set_csr_data" << std::endl;
    std::cout << "#   sparse::release_matrix_handle" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "#   std::complex<float>" << std::endl;
    std::cout << "#   std::complex<double>" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type, MatrixMultiplyExample
//  runs is with all supported data types
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {

        sycl::device my_dev;
        bool my_dev_is_found = false;
        get_sycl_device(my_dev, my_dev_is_found, *it);

        if (my_dev_is_found) {
            std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

            std::cout << "\tRunning with single precision real data type:" << std::endl;
            status = run_sparse_matrix_omatcopy_example<float, std::int32_t>(my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision real data type:" << std::endl;
                status = run_sparse_matrix_omatcopy_example<double, std::int32_t>(my_dev);
                if (status != 0)
                    return status;
            }

            std::cout << "\tRunning with single precision complex data type:" << std::endl;
            status = run_sparse_matrix_omatcopy_example<std::complex<float>, std::int32_t>(my_dev);
            if (status != 0)
                return status;

            if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                std::cout << "\tRunning with double precision complex data type:" << std::endl;
                status = run_sparse_matrix_omatcopy_example<std::complex<double>, std::int32_t>(my_dev);
                if (status != 0)
                    return status;
            }
        }
        else {
#ifdef FAIL_ON_MISSING_DEVICES
            std::cout << "No " << sycl_device_names[*it]
                      << " devices found; Fail on missing devices "
                         "is enabled.\n";
            return 1;
#else
            std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                      << sycl_device_names[*it] << " tests.\n";
#endif
        }
    }

    return status;
}
