# Copyright (C) 2023 Intel Corporation
# SPDX-License-Identifier: MIT

import re
from pathlib import Path

import numpy as np
import pytz

from mpp.parsers.data_parser import DataParser, _Line, EventPatterns, ParserAttributes
from mpp.parsers.emon.emon_system_information import EmonSystemInformationParser


class _EmonLine(_Line):
    """
    Store a line from the EMON file, extracting the relevant information as class attributes.

    Extract attributes from a line in the EMON raw file.  Each line represents the data about a given event for
    the sample duration.  Each line may have a breakdown of the event counts of the system across
    sockets/cores/threads/channels/etc.

    """

    def __init__(self, line_values, unique_os_processors, timestamp=None):
        super().__init__(line_values, unique_os_processors, timestamp)
        # Attributes must be retrieved in this order for EmonParser, because we use the pop method: name, device,
        # tsc_count, values
        self._name = self.line_values.pop(0)
        self._event_patterns = EventPatterns()
        self._device = self._set_event_device()
        self._tsc_count = float(self.line_values.pop(0).replace(',', ''))
        self._values = np.array([v.replace(',', '').replace('N/A', 'nan') for v in self.line_values], dtype=float)


class EmonParser(DataParser):
    """
    Parse EMON data file (emon.dat)
    """

    attributes = ParserAttributes('\t', "Version Info:", '%m/%d/%Y %H:%M:%S.%f',
                                         re.compile(r'(\d{2})/(\d{2})/(\d{4})\s(\d{2}):(\d{2}):(\d{2}).(\d{3})'))

    line_class = _EmonLine

    def __init__(self, input_file: Path,
                 emon_v_file: Path = None,
                 timezone: pytz.tzinfo = None,
                 ref_tsc_hz: int = 0):
        """
        Initialize the EMON parser

        :param input_file: the EMON data file to parse
        :param emon_v_file: an optional emon-v.dat file containing system information
        :param timezone: an optional timezone object for converting timestamp strings
        :param ref_tsc_hz: an optional system frequency value (in Hz). Overrides system information in the input file
                            (if such information exists)
        """
        # super().__init__ requires initialized system information
        self._system_info = EmonSystemInformationParser(emon_v_file if emon_v_file else input_file, ref_tsc_hz)
        super().__init__(input_file, timezone)

    @property
    def system_info(self):
        return self._system_info

    def convert_to_datetime_from_line(self, line: str):
        return self.convert_to_datetime(line)

    def _get_timestamps_not_found_error_class(self):
        return _EmonTimeStampsNotFoundError


class _EmonTimeStampsNotFoundError(Exception):
    def __init__(self, message="No samples labeled with timestamps found. EMON event files should be " +
                               "collected using -c <arg> to ensure all samples in the event file are timestamped, "
                               "which allows for processing."):
        super().__init__(message)
