# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: MIT

from typing import List


class DeviceType:
    CORE = 'core'
    UNCORE = 'UNCORE'
    SYSTEM = 'SYSTEM'


class Device:
    """
    Device handling class all supported devices ('core', 'bigcore', 'smallcore', 'cha', etc.)
    note: label is blank for non-hybrid cores (ex: 'core') because no core type specifier is
          added to report and filenames for non-hybrid cores.
    @param type_name: gets the type of device, used for filtering an event df
    """
    valid_device_names: List[str] = []

    def __init__(self,
                 type_name: str = DeviceType.CORE,
                 aggregation_levels: List['ViewAggregationLevel'] = None,
                 metric_computer: 'MetricComputer' = None):
        self.__type_name: str = type_name
        self.__aggregation_levels = aggregation_levels
        self.__label: str = '' if type_name == DeviceType.CORE else self.__type_name
        self.__update_uncore_labels()
        self.__exclusions = [type for type in self.valid_device_names if type != self.__type_name]
        self.__metric_computer = metric_computer

        def validate_preconditions():
            if type_name not in Device.valid_device_names:
                raise ValueError(f'{self.__type_name} is not a valid device.')

        self.__validate_preconditions()

    def __eq__(self, other):
        return self.label == other.label and self.type_name == other.type_name

    def __update_uncore_labels(self):
        if 'UNC_' in self.__label:
            self.__label = self.__label.replace('UNC_', '').lower()

    @property
    def aggregation_levels(self):
        return self.__aggregation_levels

    @property
    def metric_computer(self):
        return self.__metric_computer

    @property
    def type_name(self):
        """
        type name for this core
        """
        return self.__type_name

    @property
    def label(self):
        """
        label for this core. Used in filenames, sheet names, etc
        note: label is blank for non-hybrid cores (ex: 'core')
        """
        return self.__label

    @property
    def exclusions(self):
        """
        a list of cores types to filter out when generating reports for this core_type
        """
        return self.__exclusions

    def decorate_label(self, prefix: str = '', postfix: str = ''):
        """
        method for prefixing/postfixing the label with specified characters
        when preparing the label for use in a filename, chart name, stdout statement, etc
        :param prefix: a string to add to the beginning of the label (ex: ' ', '_'
        :param postfix: a string to add to the end of the label (ex: ' ', '_')
        :return: a copy of the decorated label
        """
        return f'{prefix}{self.__label}{postfix}' if self.__label else ''

    @staticmethod
    def set_valid_device_names(unique_devices: List[str]):
        """
        Set valid device names, a static list used for all Device objects
        :param unique_devices: unique/valid devices (example: unique devices parsed from the emon data file)
        """
        Device.valid_device_names = unique_devices

    def __validate_preconditions(self):
        if self.__type_name not in Device.valid_device_names:
            raise ValueError(f'{self.__type_name} is not a valid device.')

    def __update_uncore_labels(self):
        if 'UNC_' in self.__label:
            self.__label = self.__label.replace('UNC_', '').lower()


class DeviceCollection:
    from mpp.core.api_args import ApiArgs

    def __init__(self, api_args: ApiArgs):
        self.__args = api_args
        self.__devices = []
        self.initialize_devices()

    @property
    def devices(self):
        return self.__devices

    def initialize_devices(self):
        self.__append_core_devices()
        self.__append_uncore_devices()

    def append_device(self, device: Device):
        self.devices.append(device)

    def __append_uncore_devices(self):
        from mpp.core.views import ViewAggregationLevel
        if ViewAggregationLevel.UNCORE in self.__args.aggregation_levels:
            self._append_uncore_devices()

    def _append_uncore_devices(self):
        from mpp.core.views import ViewAggregationLevel
        uncore_devices = self.__get_uncore_device_names()
        self.__set_valid_device_names(uncore_devices)
        default_metric_computer = list(self.__args.metric_computer_map.values())[0]
        for device_name in uncore_devices:
            self.append_device(Device(device_name, [ViewAggregationLevel.UNCORE], default_metric_computer))

    def __set_valid_device_names(self, uncore_devices):
        core_devices = list(self.__args.system_information.unique_core_types.copy())
        Device.set_valid_device_names(core_devices + uncore_devices)

    def __get_uncore_device_names(self):
        from mpp.core.types import RawDataFrameColumns as rdc
        event_info = self.__args.event_info
        return list(event_info.loc[event_info[rdc.DEVICE].str.startswith('UNC_'), rdc.DEVICE].unique())

    def __append_core_devices(self):
        for core_type, metric_computer in self.__args.metric_computer_map.items():
            self.append_device(Device(core_type, self.__args.aggregation_levels, metric_computer))
