/*
 * Copyright (C) 2005 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

#include "lwpmudrv_defines.h"

#include "lwpmudrv_types.h"
#include "rise_errors.h"
#include "lwpmudrv_ecb.h"
#include "lwpmudrv_struct.h"
#include "lwpmudrv_ioctl.h"
#include "lwpmudrv.h"
#include "control.h"
#include "utility.h"
#include "output.h"

#include "fbsd_os.h"

#include <sys/proc.h>
#include <sys/types.h>
#include <machine/pcb.h>
#include <vm/vm.h>
#include <vm/pmap.h>
#include <machine/pmap.h>
#include <vm/vm_map.h>
#include <vm/vm_object.h>

#include <vm/vm_extern.h>
#include <sys/param.h>
#include <sys/vnode.h>
#include <sys/eventhandler.h>
#include <sys/mount.h>

#if __FreeBSD_version >= 1000030
#include <sys/rwlock.h>
#include <vm/vm_object.h>
#include <fs/tmpfs/tmpfs.h>
#define SEP_VM_OBJECT_LOCK	VM_OBJECT_RLOCK
#define SEP_VM_OBJECT_UNLOCK	VM_OBJECT_RUNLOCK
#else
#define SEP_VM_OBJECT_LOCK	VM_OBJECT_LOCK
#define SEP_VM_OBJECT_UNLOCK	VM_OBJECT_UNLOCK
#endif

extern uid_t          uid;
extern volatile pid_t control_pid;
static volatile S32   hooks_installed = 0;

extern int
LWPMUDRV_Abnormal_Terminate(void);

#define SHARED_REGION_64_BASE	0x800000000
#define SHARED_REGION_32_BASE	0x20000000

#define MY_TASK  PROFILE_TASK_EXIT
#define MY_UNMAP PROFILE_MUNMAP

#if defined(DRV_IA32)
static U16
fbsd_os_Get_Exec_Mode (
    struct proc *p
)
{
    return ((unsigned short) MODE_32BIT);
}
#endif

#if defined(DRV_EM64T)
static U16
fbsd_os_Get_Exec_Mode (
    struct proc *p
)
{
    if (!p) {
        return MODE_UNKNOWN;
    }
    /* make sure kstack (where td_pcb is stored) is resident */
    if ((p->p_flag & P_INMEM) == 0) {
	return (MODE_UNKNOWN);
    }

    if (FIRST_THREAD_IN_PROC(p)->td_pcb->pcb_flags & PCB_32BIT) {
        return ((unsigned short) MODE_32BIT);
    }
    return ((unsigned short) MODE_64BIT);
}
#endif

static S32
fbsd_os_Load_Image_Notify_Routine (
    char           *name,
    PVOID           base,
    U32             size,
    PVOID           offset,
    U32             pid,
    U32             parent_pid,
    U32             options,
    unsigned short  mode,
    S32             load_event,
    U32             in_notification
)
{
    char          *raw_path;
    ModuleRecord  *mra;
    char           buf[sizeof(ModuleRecord) + MAXNAMELEN + 32];
    U64            tsc_read;
    S32            local_load_event = (load_event==-1) ? 0 : load_event;

    SEP_DRV_LOG_NOTIFICATION_TRACE_IN(in_notification, "Pid: %u, ppid: %u, load_event: %d.", pid, parent_pid, load_event);
    SEP_DRV_LOG_NOTIFICATION_TRACE(in_notification, "Pid: %u: '%s'.", pid, name);

    mra = (ModuleRecord *) buf;
    memset(mra, '\0', sizeof(buf));
    raw_path = (char*) mra + sizeof(ModuleRecord);

    MODULE_RECORD_processed(mra)                   = 0;
    MODULE_RECORD_segment_type(mra)                = mode;
    MODULE_RECORD_load_addr64(mra)                 = (U64)(size_t)base;
    MODULE_RECORD_length64(mra)                    = size;
    MODULE_RECORD_segment_number(mra)              = 1; // for user modules
    MODULE_RECORD_global_module_tb5(mra)           = options & LOPTS_GLOBAL_MODULE;
    MODULE_RECORD_first_module_rec_in_process(mra) = options & LOPTS_1ST_MODREC;
    MODULE_RECORD_tsc_used(mra)                    = 1;
    MODULE_RECORD_exe(mra)                         = 0;
    MODULE_RECORD_parent_pid(mra)                  = parent_pid;
    MR_page_offset_Set(mra, (U64)offset);

    UTILITY_Read_TSC(&tsc_read);
    preempt_disable();
    tsc_read -= TSC_SKEW(CONTROL_THIS_CPU());
    preempt_enable();

    if (local_load_event) {
        MR_unloadTscSet(mra, tsc_read);
    }
    else {
        MR_unloadTscSet(mra, (U64)(-1));
    }
    MODULE_RECORD_pid_rec_index(mra)     = pid;
    MODULE_RECORD_pid_rec_index_raw(mra) = 1; // raw pid

    strncpy(raw_path, name, MAXNAMELEN);
    raw_path[MAXNAMELEN]              = 0;
    MODULE_RECORD_path_length(mra)    =  (U16) strlen(raw_path) + 1;
    MODULE_RECORD_rec_length(mra)     =  (U16) ALIGN_8(sizeof (ModuleRecord) +
                                                       MODULE_RECORD_path_length(mra));

#if defined(DRV_IA32)
    MODULE_RECORD_selector(mra)       = (pid==0) ? __KERNEL_CS : __USER_CS;
#endif
#if defined(DRV_EM64T)
    if (mode == MODE_64BIT) {
        MODULE_RECORD_selector(mra) =
            (pid==0) ? GSEL(GCODE_SEL, SEL_KPL) : GSEL(GUCODE_SEL, SEL_UPL);
    }
    else if (mode == MODE_32BIT) {
        MODULE_RECORD_selector(mra) =
            (pid==0) ? GSEL(GCODE_SEL, SEL_KPL) : GSEL(GUCODE32_SEL, SEL_UPL);
    }
#endif

    if (LOPTS_EXE & options) {
        MODULE_RECORD_exe(mra) = 1;
    }

    OUTPUT_Module_Fill((PVOID)mra, MODULE_RECORD_rec_length(mra), in_notification);

    SEP_DRV_LOG_NOTIFICATION_TRACE_OUT(in_notification, "OS_SUCCESS.");
    return OS_SUCCESS;
}

//
// Register the module for a process.  The task_struct and mm
// should be locked if necessary to make sure they don't change while we're
// iterating...
// Used as a service routine
//
/*
static S32
fbsd_os_VMA_For_Process (
    struct proc           *p,
    struct vnode          *vp,
    S32                    load_event,
    U32                   *first
)
{
    U32  options;
    char *pname = "(N/A)";
    char *freebuf = NULL;
    U32  ppid  = 0;

    vn_fullpath(curthread, vp, &pname, &freebuf);

    options = 0;
    // if the VM_PROT_EXECUTE flag is set then this is the module
    // that is being used to name the module
    //if (vma->protection & VM_PROT_EXECUTE) {
    //    options |= LOPTS_EXE;
    //}
    // mark the first of the bunch...
    if (*first == 1) {
        options |= LOPTS_1ST_MODREC;
        *first = 0;
    }

    if (p && p->p_pptr) {
        ppid = p->p_pptr->p_pid;
    }

    // record this module
    fbsd_os_Load_Image_Notify_Routine(pname,
                                      (PVOID)vma->start,
                                      (vma->end - vma->start),
                                      (PVOID)vma->offset,
                                      p->p_pid,
                                      ppid,
                                      options,
                                      fbsd_os_Get_Exec_Mode(p),
                                      load_event);

    if (freebuf)
        free(freebuf, M_TEMP);
    return OS_SUCCESS;
}
*/
//
// Common loop to enumerate all modules for a process.  The task_struct and mm
// should be locked if necessary to make sure they don't change while we're
// iterating...
//
static S32
fbsd_os_Enum_Modules_For_Process (
    struct proc        *p,
    S32                 load_event,
    U32                 in_notification
)
{
    struct vmspace *vm;
    vm_map_entry_t  cur;
    vm_object_t     object;
    struct vnode   *vp;
    U32              	first = 1, exe_found = 0, options;
    char		*pname, *freebuf;
    U32			ppid = 0;
    int			error;
    U16			mode;
#if __FreeBSD_version < 1000021
    int			vfslocked;
#endif

#if defined(SECURE_SEP)
    uid_t                  l_uid;

    SEP_DRV_LOG_NOTIFICATION_TRACE_IN(in_notification, "P: %p, load_event: %d, in_notif: %u.", p, load_event, in_notification);

    l_uid = DRV_GET_UID(p);

    /*
     * Check for:  same uid, or root uid
     */
    if (l_uid != uid && l_uid != 0) {
        SEP_DRV_LOG_NOTIFICATION_TRACE_OUT(in_notification, "OS_SUCCESS (l_uid != uid && l_uid).");
        return OS_SUCCESS;
    }
#endif

    mode = fbsd_os_Get_Exec_Mode(p);

    vm = vmspace_acquire_ref(p);

    if (vm == NULL) {
        goto load_image_notify;
    }

    vm_map_lock_read(&vm->vm_map);
#if __FreeBSD_version < 1300062
    for (cur = vm->vm_map.header.next;
         cur != &vm->vm_map.header;
         cur = cur->next) {
#else
    VM_MAP_ENTRY_FOREACH(cur, &vm->vm_map) {
#endif
        object = cur->object.vm_object;

        if (object == NULL)
            continue;

        SEP_VM_OBJECT_LOCK(object);
#if __FreeBSD_version > 1000021
    if (!(cur->protection & VM_PROT_EXECUTE) ||
            (cur->eflags & MAP_ENTRY_IS_SUB_MAP) ||
            (object->type != OBJT_VNODE && (object->type != OBJT_SWAP || (object->flags & OBJ_TMPFS) == 0))) {
        SEP_VM_OBJECT_UNLOCK(object);
        continue;
    }
#else
	if (!(cur->protection & VM_PROT_EXECUTE) ||
			(cur->eflags & MAP_ENTRY_IS_SUB_MAP) ||
			(object->type != OBJT_VNODE)) {
		SEP_VM_OBJECT_UNLOCK(object);
		continue;
	}
#endif
#if __FreeBSD_version > 1000021
	if (object->type == OBJT_VNODE){
		vp = object->handle;
	} else if (object->type == OBJT_SWAP) {
#if __FreeBSD_version > 1301510
		vp = VM_TO_TMPFS_VP(object);
#else
		vp = object->un_pager.swp.swp_tmpfs;
#endif
	}
#else
	vp = object->handle;

#endif
	vref(vp);
	SEP_VM_OBJECT_UNLOCK(object);
#if __FreeBSD_version < 1300062
	error = vn_fullpath(curthread, vp, &pname, &freebuf);
#else
	error = vn_fullpath(vp, &pname, &freebuf);
#endif

#if __FreeBSD_version < 1000021
        vfslocked = VFS_LOCK_GIANT(vp->v_mount);
        vrele(vp);
        VFS_UNLOCK_GIANT(vfslocked);
#else
	vrele(vp);
#endif

	options = 0;

	/*
	 * We don't have a *good* way of determining when a mapping is an
	 *  executable or a shared library.  On 10.x, it seems shared
	 *  libraries are always mapped starting at 0x800000000 for 64-bit
	 *  processes, and 0x20000000 for 32-bit processes, so just use that
	 *  for now.
	 */
	if ((mode == MODE_64BIT && cur->start < SHARED_REGION_64_BASE) ||
	    (mode == MODE_32BIT && cur->start < SHARED_REGION_32_BASE)) {
		if (!exe_found) {
			options |= LOPTS_EXE;
			exe_found = 0;
		}
	}

        // mark the first of the bunch...
        if (first == 1) {
            options |= LOPTS_1ST_MODREC;
            first = 0;
        }

        if (p && p->p_pptr)
            ppid = p->p_pptr->p_pid;

        // record this module
        fbsd_os_Load_Image_Notify_Routine(error == 0 ? pname : "(N/A)",
                                      (PVOID)cur->start,
                                      (cur->end - cur->start),
                                      (PVOID)cur->offset,
                                      p->p_pid,
                                      ppid,
                                      options,
                                      mode,
                                      load_event,
                                      in_notification);

        if (!error)
            free(freebuf, M_TEMP);
    }
    vm_map_unlock_read(&vm->vm_map);
    vmspace_free(vm);

load_image_notify:

    if (first == 1) {
        // No modules found for this process.  This will
        //  happen for kernel processes.
        fbsd_os_Load_Image_Notify_Routine(p->p_comm,
                                          NULL,
                                          0,
                                          0,
                                          p->p_pid,
                                          (p->p_pptr) ? p->p_pptr->p_pid : 0,
                                          LOPTS_EXE | LOPTS_1ST_MODREC,
                                          mode,
                                          1,
                                          in_notification);
    }

    SEP_DRV_LOG_NOTIFICATION_TRACE_OUT(in_notification, "OS_SUCCESS.");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          OS_STATUS FBSD_OS_Enum_Process_Modules(DRV_BOOL at_end)
 *
 * @brief       gather all the process modules that are present.
 *
 * @param       at_end - the collection happens at the end of the sampling run
 *
 * @return      OS_SUCCESS
 *
 * <I>Special Notes:</I>
 *              This routine gathers all the process modules that are present
 *              in the system at this time.  If at_end is set to be TRUE, then
 *              act as if all the modules are being unloaded.
 *
 */
extern OS_STATUS
FBSD_OS_Enum_Process_Modules (
    DRV_BOOL  at_end
)
{
    int                 n = 0;
    struct proc        *p;

    SEP_DRV_LOG_TRACE_IN("Enum_Process_Modules begin tasks.");

    if (GET_DRIVER_STATE() == DRV_STATE_TERMINATING) {
        SEP_DRV_LOG_TRACE_OUT("Early return: TERMINATING.");
        return OS_SUCCESS;
    }

    sx_slock(&allproc_lock);
    FOREACH_PROC_IN_SYSTEM(p){
        SEP_DRV_LOG_TRACE("Enum_Process_Modules looking at task %d.", n);
        /*
         *  Call driver notification routine for each module
         *  that is mapped into the process created by the fork
         */
        fbsd_os_Enum_Modules_For_Process(p, at_end?-1:0, !SEP_IN_NOTIFICATION);
        n++;
    }
    sx_sunlock(&allproc_lock);
    SEP_DRV_LOG_TRACE_OUT("Enum_Process_Modules done with %d tasks.", n);

    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          static int fbsd_os_Exit_Task_Notify(struct notifier_block * self,
 *                  unsigned long val, PVOID data)
 * @brief       this function is called whenever a task exits
 *
 * @param       self IN  - not used
 *              val IN  - not used
 *              data IN  - this is cast into the task_struct of the exiting task
 *
 * @return      none
 *
 * <I>Special Notes:</I>
 * this function is called whenever a task exits.  It is called right before
 * the virtual memory areas are freed.  We just enumerate through all the modules
 * of the task and set the unload sample count and the load event flag to 1 to
 * indicate this is a module unload
 */
eventhandler_tag process_exit_tag;

static void
sep_process_exit(void *arg, struct proc *p)
{
    SEP_DRV_LOG_NOTIFICATION_IN("Pid: %u, p: %p.", p->p_pid, p); // exceptionnally dereferencing a pointer in IN message as
                                                                 // a) p should never be NULL
                                                                 // b) p is provided by the OS: if the address is corrupted, we have a bigger problem at hand
                                                                 // c) this makes the message a lot more useful when TRACE is not enabled

    if (GET_DRIVER_STATE() == DRV_STATE_UNINITIALIZED) {
        SEP_DRV_LOG_NOTIFICATION_WARNING(SEP_IN_NOTIFICATION, "Received exit notification in UNINITIALIZED state!");
        goto clean_return;
    }

    if (p->p_pid == control_pid) {
        SEP_DRV_LOG_NOTIFICATION_WARNING(SEP_IN_NOTIFICATION, "The collector task has been terminated via an uncatchable signal.");
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
        goto clean_return;
    }

    if (!p->p_vmspace) {
        SEP_DRV_LOG_NOTIFICATION_ERROR(SEP_IN_NOTIFICATION, "P_vmspace is NULL!");
        goto clean_return;
    }

    if (GET_DRIVER_STATE() != DRV_STATE_TERMINATING) {
        fbsd_os_Enum_Modules_For_Process(p, 1, SEP_IN_NOTIFICATION);
    }

    clean_return:
    SEP_DRV_LOG_NOTIFICATION_OUT("");
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          int FBSD_OS_Install_Hooks(VOID)
 * @brief       registers the profiling callbacks
 *
 * @param       none
 *
 * @return      0 for success everything else fails
 *
 * <I>Special Notes:</I>
 *
 * None
 */


extern VOID
FBSD_OS_Install_Hooks (
    VOID
)
{
    SEP_DRV_LOG_INIT_IN("Installing OS hooks...");

    if (hooks_installed == 1) {
        SEP_DRV_LOG_INIT_OUT("The OS Hooks are already installed.");
        return;
    }

    process_exit_tag = EVENTHANDLER_REGISTER(process_exit, sep_process_exit, NULL, EVENTHANDLER_PRI_ANY);

    hooks_installed = 1;

    SEP_DRV_LOG_INIT_OUT("");
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID FBSD_OS_Uninstall_Hooks(VOID)
 * @brief       unregisters the profiling callbacks
 *
 * @param       none
 *
 * @return
 *
 * <I>Special Notes:</I>
 *
 * None
 */
extern VOID
FBSD_OS_Uninstall_Hooks (
    VOID
)
{
    SEP_DRV_LOG_INIT_IN("Uninstalling OS Hooks...");

    if (hooks_installed == 0) {
        SEP_DRV_LOG_INIT_OUT("Hooks are not installed!");
        return;
    }


    hooks_installed = 0;
    EVENTHANDLER_DEREGISTER(process_exit, process_exit_tag);

    SEP_DRV_LOG_INIT_OUT("");
}
