/*
 * Copyright (C) 2005 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

#include "lwpmudrv_defines.h"

#include "lwpmudrv_types.h"
#include "rise_errors.h"
#include "lwpmudrv_ecb.h"
#include "lwpmudrv.h"
#include "pci.h"
#include "utility.h"
#include "control.h"

#include <sys/bus.h>
#include <dev/pci/pcireg.h>
#include <dev/pci/pcivar.h>


/* ------------------------------------------------------------------------- */
/*!
 * @fn extern VOID PCI_Initialize(VOID)
 *
 * @param   none
 *
 * @return  none
 *
 * @brief   Initializes PCI bus accesses.
 *
 */
extern VOID PCI_Initialize (
    VOID
)
{
    U32 bus, devn, func, i;
    device_t dev;
    U32 num_found_buses = 0;
    U32 prev_val = 0;

    SEP_DRV_LOG_INIT_IN("Initializing pci_buses...");

    for (i = 0; i < num_packages; i++) {
        prev_val = num_found_buses;
        for (bus = 0; bus < MAX_BUSNO; bus++) {
            for (devn = 0; devn < MAX_PCI_DEVNO; devn++) {
                for (func = 0; func < MAX_PCI_FUNCNO; func++) {
                    dev = pci_find_dbsf(i, bus, devn, func);
                    if (dev) {
                        SEP_DRV_LOG_DETECTION("Found PCI domain 0x%x, bus 0x%x at %p.", i, bus, dev);
                        num_found_buses++;
                    }
                    SEP_DRV_LOG_TRACE("pci_buses domain %u, bus %u: %p.", i, bus, dev);
                }
            }
        }
        if (prev_val < num_found_buses) {
            num_pci_domains++;
        }
    }

    if (!num_pci_domains) {
        num_pci_domains = 1;
    }

    SEP_DRV_LOG_INIT_OUT("Found %u buses across %u domains.", num_found_buses, num_pci_domains);
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn extern U32 PCI_Read_U32(domain, bus, device, function, offset)
 *
 * @param    domain     - target domain
 * @param    bus        - target bus
 * @param    device     - target device
 * @param    function   - target function
 * @param    offset     - target register offset
 *
 * @return  Value at this location
 *
 * @brief   Reads a U32 from PCI configuration space
 *
 */
extern U32 PCI_Read_U32 (
    U32    domain,
    U32    bus,
    U32    device,
    U32    function,
    U32    offset
)
{
    U32      res   = 0;
    device_t dev;

    SEP_DRV_LOG_REGISTER_IN("Will read Domain %x BDF(%x:%x:%x)[0x%x](4B)...", domain, bus, device, function, offset);

    if (bus < MAX_BUSNO && (dev = pci_find_dbsf(domain, bus, device, function))) {
        res = pci_read_config(dev, offset, 4);
    }
    else {
        SEP_DRV_LOG_ERROR("Could not read Domain %x BDF(%x:%x:%x)[0x%x]: bus not found!", domain, bus, device, function, offset);
    }

    SEP_DRV_LOG_REGISTER_OUT("Has read Domain %x BDF(%x:%x:%x)[0x%x](4B): 0x%x.", domain, bus, device, function, offset, res);
    return res;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn extern U32 PCI_Read_U32_Valid(domain, bus, device, function, offset, invalid_value)
 *
 * @param    domain         - target domain
 * @param    bus            - target bus
 * @param    device         - target device
 * @param    function       - target function
 * @param    offset         - target register offset
 * @param    invalid_value  - value against which to compare the PCI-obtained value
 *
 * @return  Value at this location (if value != invalid_value), 0 otherwise
 *
 * @brief   Reads a U32 from PCI configuration space
 *
 */
extern U32 PCI_Read_U32_Valid (
    U32    domain,
    U32    bus,
    U32    device,
    U32    function,
    U32    offset,
    U32    invalid_value
)
{
    U32      res   = 0;
    device_t dev;

    SEP_DRV_LOG_REGISTER_IN("Will read Domain%x BDF(%x:%x:%x)[0x%x](4B)...", domain, bus, device, function, offset);

    if (bus < MAX_BUSNO && (dev = pci_find_dbsf(domain, bus, device, function))) {
        res = pci_read_config(dev, offset, 4);
        if (res == invalid_value) {
            res = 0;
            SEP_DRV_LOG_REGISTER_OUT("Has read Domain%x BDF(%x:%x:%x)[0x%x](4B): 0 (invalid value).", domain, bus, device, function, offset, res);
        }
        else {
            SEP_DRV_LOG_REGISTER_OUT("Has read Domain%x BDF(%x:%x:%x)[0x%x](4B): 0x%x.", domain, bus, device, function, offset, res);
        }
    }
    else {
        SEP_DRV_LOG_REGISTER_OUT("Could not read Domain%x BDF(%x:%x:%x)[0x%x]: bus not found!", domain, bus, device, function, offset);
    }

    return res;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn extern U32 PCI_Read_U64(domain, bus, device, function, offset)
 *
 * @param   domain     - target domain
 * @param   bus        - target bus
 * @param   device     - target device
 * @param   function   - target function
 * @param   offset     - target register offset
 *
 * @return  Value at this location
 *
 * @brief   Reads a U64 from PCI configuration space
 *
 */
extern U64 PCI_Read_U64 (
    U32    domain,
    U32    bus,
    U32    device,
    U32    function,
    U32    offset
)
{
    U64      res   = 0;
    device_t dev;

    SEP_DRV_LOG_REGISTER_IN("Will read Domain%x BDF(%x:%x:%x)[0x%x](8B)...", domain, bus, device, function, offset);

    if (bus < MAX_BUSNO && (dev = pci_find_dbsf(domain, bus, device, function))) {
        res = pci_read_config(dev, offset, 4) |
              ((U64)pci_read_config(dev, offset + 4, 4) << 32);
    }
    else {
        SEP_DRV_LOG_ERROR("Could not read Domain%x BDF(%x:%x:%x)[0x%x]: bus not found!", domain, bus, device, function, offset);
    }

    SEP_DRV_LOG_REGISTER_OUT("Has read Domain%x BDF(%x:%x:%x)[0x%x](8B): 0x%llx.", domain, bus, device, function, offset, res);
    return res;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn extern U32 PCI_Read_U64_Valid(domain, bus, device, function, offset, invalid_value)
 *
 * @param   domain          - target domain
 * @param   bus             - target bus
 * @param   device          - target device
 * @param   function        - target function
 * @param   offset          - target register offset
 * @param   invalid_value   - value against which to compare the PCI-obtained value
 *
 * @return  Value at this location (if value != invalid_value), 0 otherwise
 *
 * @brief   Reads a U64 from PCI configuration space
 *
 */
extern U64 PCI_Read_U64_Valid (
    U32    domain,
    U32    bus,
    U32    device,
    U32    function,
    U32    offset,
    U64    invalid_value
)
{
    U64      res   = 0;
    device_t dev;

    SEP_DRV_LOG_REGISTER_IN("Will read Domain%x BDF(%x:%x:%x)[0x%x](8B)...", domain, bus, device, function, offset);

    if (bus < MAX_BUSNO && (dev = pci_find_dbsf(domain, bus, device, function))) {
        res = pci_read_config(dev, offset, 4) |
              ((U64)pci_read_config(dev, offset + 4, 4) << 32);

        if (res == invalid_value) {
            res = 0;
            SEP_DRV_LOG_REGISTER_OUT("Has read Domain%x BDF(%x:%x:%x)[0x%x](8B): 0x%llx (invalid value).", domain, bus, device, function, offset, res);
        }
        else {
            SEP_DRV_LOG_REGISTER_OUT("Has read Domain%x BDF(%x:%x:%x)[0x%x](8B): 0x%llx.", domain, bus, device, function, offset, res);
        }
    }
    else {
        SEP_DRV_LOG_REGISTER_OUT("Could not read Domain%x BDF(%x:%x:%x)[0x%x]: bus not found!", domain, bus, device, function, offset);
    }

    return res;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn extern U32 PCI_Write_U32(domain, bus, device, function, offset, value)
 *
 * @param    domain         - target domain
 * @param    bus            - target bus
 * @param    device         - target device
 * @param    function       - target function
 * @param    offset         - target register offset
 * @param    value          - value to write
 *
 * @return  0 in case of success, 1 otherwise
 *
 * @brief    Writes a U32 to PCI configuration space
 *
 */
extern U32 PCI_Write_U32 (
    U32    domain,
    U32    bus,
    U32    device,
    U32    function,
    U32    offset,
    U32    value
)
{
    U64      res   = 0;
    device_t dev;

    SEP_DRV_LOG_REGISTER_IN("Will write Domain%x BDF(%x:%x:%x)[0x%x](4B): 0x%x...", domain, bus, device, function, offset, value);

    if (bus < MAX_BUSNO && (dev = pci_find_dbsf(domain, bus, device, function))) {
        pci_write_config(dev, offset, value, 4);
        SEP_DRV_LOG_REGISTER_OUT("Has written Domain%x BDF(%x:%x:%x)[0x%x](4B): 0x%x.", domain, bus, device, function, offset, value);
    }
    else {
        SEP_DRV_LOG_ERROR("Could not write Domain%x BDF(%x:%x:%x)[0x%x]: bus not found!", domain, bus, device, function, offset);
        res = 1;
        SEP_DRV_LOG_REGISTER_OUT("Failed to write Domain%x BDF(%x:%x:%x)[0x%x](4B): 0x%x.", domain, bus, device, function, offset, value);
    }

    return res;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn extern U32 PCI_Write_U64(domain, bus, device, function, offset, value)
 *
 * @param    domain         - target domain
 * @param    bus            - target bus
 * @param    device         - target device
 * @param    function       - target function
 * @param    offset         - target register offset
 * @param    value          - value to write
 *
 * @return  0 in case of success, 1 otherwise
 *
 * @brief    Writes a U64 to PCI configuration space
 *
 */
extern U32 PCI_Write_U64 (
    U32    domain,
    U32    bus,
    U32    device,
    U32    function,
    U32    offset,
    U64    value
)
{
    U64      res   = 0;
    device_t dev;

    SEP_DRV_LOG_REGISTER_IN("Will write Domain%x BDF(%x:%x:%x)[0x%x](8B): 0x%llx...", domain, bus, device, function, offset, value);

    if (bus < MAX_BUSNO && (dev = pci_find_dbsf(domain, bus, device, function))) {
        pci_write_config(dev, offset    , (U32)value, 4);
        pci_write_config(dev, offset + 4, (U32)(value >> 32), 4);
        SEP_DRV_LOG_REGISTER_OUT("Has written Domain%x BDF(%x:%x:%x)[0x%x](8B): 0x%llx.", domain, bus, device, function, offset, value);
    }
    else {
        SEP_DRV_LOG_ERROR("Could not write Domain%x BDF(%x:%x:%x)[0x%x]: bus not found!", domain, bus, device, function, offset);
        res = 1;
        SEP_DRV_LOG_REGISTER_OUT("Failed to write Domain%x BDF(%x:%x:%x)[0x%x](8B): 0x%llx.", domain, bus, device, function, offset, value);
    }

    return res;
}




/* ------------------------------------------------------------------------- */
/*!
 * @fn extern int PCI_Read_From_Memory_Address(addr, val)
 *
 * @param    addr    - physical address in mmio
 * @param   *value  - value at this address
 *
 * @return  status
 *
 * @brief   Read memory mapped i/o physical location
 *
 */
extern int
PCI_Read_From_Memory_Address (
    U32 addr,
    U32* val
)
{
    U32 aligned_addr, offset, value;
    vm_offset_t base;
    OS_STATUS result = OS_SUCCESS;

    SEP_DRV_LOG_REGISTER_IN("Will read physical memory at 0x%x(4B) into %p.", addr, val);

    if (addr == 0) {
        SEP_DRV_LOG_ERROR("Invalid address: 0x%x!", addr);
        result = OS_INVALID;
    }
    else {
        offset       = addr & ~PAGE_MASK;
        aligned_addr = addr & PAGE_MASK;

        base = (vm_offset_t)pmap_mapdev(aligned_addr, PAGE_SIZE);
        if (base == 0) {
            SEP_DRV_LOG_ERROR("Memory mapping failure for 0x%x!", addr);
            result = OS_INVALID;
        }
        else {
            value = readl(base+offset);
            *val = value;
#if __FreeBSD_version >= 1400000
            pmap_unmapdev((void *)base, PAGE_SIZE);
#else
            pmap_unmapdev(base, PAGE_SIZE);
#endif
        }
    }

    if (result == OS_SUCCESS) {
        SEP_DRV_LOG_REGISTER_OUT("Has read physical memory at 0x%x(4B): 0x%x.", addr, *val);
    }
    else {
        *val = 0;
        SEP_DRV_LOG_REGISTER_OUT("Failed to read physical memory at 0x%x(4B).", addr);
    }
    return result;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn extern int PCI_Write_To_Memory_Address(addr, val)
 *
 * @param   addr   - physical address in mmio
 * @param   value  - value to be written
 *
 * @return  status
 *
 * @brief   Write to memory mapped i/o physical location
 *
 */
extern int
PCI_Write_To_Memory_Address (
    U32 addr,
    U32 val
)
{
    U32 aligned_addr, offset;
    vm_offset_t base;
    OS_STATUS result = OS_SUCCESS;

    SEP_DRV_LOG_REGISTER_IN("Will write physical memory at %p(4B): 0x%x.", addr, val);

    if (addr == 0) {
        SEP_DRV_LOG_ERROR("Invalid address: 0x%x!", addr);
        result = OS_INVALID;
    }
    else {
        offset       = addr & ~PAGE_MASK;
        aligned_addr = addr & PAGE_MASK;

        base = (vm_offset_t)pmap_mapdev(aligned_addr, PAGE_SIZE);
        if (base == 0) {
            SEP_DRV_LOG_ERROR("Memory mapping failure for 0x%x!", addr);
            result = OS_INVALID;
        }

        writel(base+offset,val);

#if __FreeBSD_version >= 1400000
        pmap_unmapdev((void *)base, PAGE_SIZE);
#else
        pmap_unmapdev(base, PAGE_SIZE);
#endif
    }

    if (result == OS_SUCCESS) {
        SEP_DRV_LOG_REGISTER_OUT("Has written physical memory at 0x%x(4B): 0x%x.", addr, val);
    }
    else {
        SEP_DRV_LOG_REGISTER_OUT("Failed to write physical memory at 0x%x(4B) (0x%x).", addr, val);
    }
    return result;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn extern int PCI_Read_From_Memory_Address_U64(addr, val)
 *
 * @param    addr    - physical address in mmio
 * @param   *value  - value at this address
 *
 * @return  status
 *
 * @brief   Read memory mapped i/o physical location
 *
 */
extern int
PCI_Read_From_Memory_Address_U64 (
    U64 addr,
    U64* val
)
{
    U64 aligned_addr, offset, value;
    vm_offset_t base;
    OS_STATUS result = OS_SUCCESS;

    SEP_DRV_LOG_REGISTER_IN("Will read physical memory at 0x%llx(4B) into %p.", addr, val);

    if (addr == 0) {
        SEP_DRV_LOG_ERROR("Invalid address: 0x%llx!", addr);
        result = OS_INVALID;
    }
    else {
        offset       = addr & ~PAGE_MASK;
        aligned_addr = addr & PAGE_MASK;

        base = (vm_offset_t)pmap_mapdev(aligned_addr, PAGE_SIZE);
        if (base == 0) {
            SEP_DRV_LOG_ERROR("Memory mapping failure for 0x%llx!", addr);
            result = OS_INVALID;
        }
        else {
            value = readl(base+offset);
            *val = value;
#if __FreeBSD_version >= 1400000
            pmap_unmapdev((void *)base, PAGE_SIZE);
#else
            pmap_unmapdev(base, PAGE_SIZE);
#endif
        }
    }

    if (result == OS_SUCCESS) {
        SEP_DRV_LOG_REGISTER_OUT("Has read physical memory at 0x%llx(4B): 0x%llx.", addr, *val);
    }
    else {
        *val = 0;
        SEP_DRV_LOG_REGISTER_OUT("Failed to read physical memory at 0x%llx(4B).", addr);
    }
    return result;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn extern int PCI_Write_To_Memory_Address_U64(addr, val)
 *
 * @param   addr   - physical address in mmio
 * @param   value  - value to be written
 *
 * @return  status
 *
 * @brief   Write to memory mapped i/o physical location
 *
 */
extern int
PCI_Write_To_Memory_Address_U64 (
    U64 addr,
    U64 val
)
{
    U64 aligned_addr, offset;
    vm_offset_t base;
    OS_STATUS result = OS_SUCCESS;

    SEP_DRV_LOG_REGISTER_IN("Will write physical memory at %p(4B): 0x%llx.", addr, val);

    if (addr == 0) {
        SEP_DRV_LOG_ERROR("Invalid address: 0x%x!", addr);
        result = OS_INVALID;
    }
    else {
        offset       = addr & ~PAGE_MASK;
        aligned_addr = addr & PAGE_MASK;

        base = (vm_offset_t)pmap_mapdev(aligned_addr, PAGE_SIZE);
        if (base == 0) {
            SEP_DRV_LOG_ERROR("Memory mapping failure for 0x%llx!", addr);
            result = OS_INVALID;
        }

        writel(base+offset,val);

#if __FreeBSD_version >= 1400000
        pmap_unmapdev((void *)base, PAGE_SIZE);
#else
        pmap_unmapdev(base, PAGE_SIZE);
#endif
    }

    if (result == OS_SUCCESS) {
        SEP_DRV_LOG_REGISTER_OUT("Has written physical memory at 0x%llx(4B): 0x%llx.", addr, val);
    }
    else {
        SEP_DRV_LOG_REGISTER_OUT("Failed to write physical memory at 0x%llx(4B) (0x%llx).", addr, val);
    }
    return result;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn extern U32 PCI_Map_Memory(SEP_MMIO_NODE *node, U64 phy_address, U64 map_size)
 *
 * @param    node        - MAP NODE to use
 * @param    phy_address - Address to be mapped
 * @param    map_size    - Amount of memory to map (has to be a multiple of 4k)
 *
 * @return   OS_SUCCESS or OS_INVALID
 *
 * @brief    Maps a physical address to a virtual address
 *
 */
extern OS_STATUS
PCI_Map_Memory (
    SEP_MMIO_NODE *node,
    U64           phy_address,
    U32           map_size
)
{
    U8  *res;

    SEP_DRV_LOG_INIT_IN("Node: %p, phy_address: %llx, map_size: %u.", node, phy_address, map_size);

    if (!node              ||
        !phy_address       ||
        !map_size) {
        SEP_DRV_LOG_ERROR_INIT_OUT("Invalid parameters, aborting!");
        return OS_INVALID;
    }

    res = pmap_mapdev(phy_address, map_size);
    if (!res) {
        SEP_DRV_LOG_ERROR_INIT_OUT("Map operation failed!");
        return OS_INVALID;
    }

    SEP_MMIO_NODE_physical_address(node) = (UIOP)phy_address;
    SEP_MMIO_NODE_virtual_address(node)  = (UIOP)res;
    SEP_MMIO_NODE_map_token(node)        = (UIOP)res;
    SEP_MMIO_NODE_size(node)             = map_size;

    SEP_DRV_LOG_INIT_OUT("Addr:0x%llx->0x%llx,tok:0x%llx,sz:%u.",
        SEP_MMIO_NODE_physical_address(node),
        SEP_MMIO_NODE_virtual_address(node),
        SEP_MMIO_NODE_map_token(node),
        SEP_MMIO_NODE_size(node));
    return OS_SUCCESS;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn extern void PCI_Unmap_Memory(SEP_MMIO_NODE *node)
 *
 * @param   node - memory map node to clean up
 *
 * @return  none
 *
 * @brief   Unmaps previously mapped memory
 *
 */
extern void
PCI_Unmap_Memory (
    SEP_MMIO_NODE *node
)
{
    SEP_DRV_LOG_INIT_IN("Unmapping node %p.", node);

    if (node) {
        if (SEP_MMIO_NODE_size(node)) {
            SEP_DRV_LOG_TRACE("Unmapping token 0x%llx (0x%llx->0x%llx)[%uB].",
                SEP_MMIO_NODE_map_token(node),
                SEP_MMIO_NODE_physical_address(node),
                SEP_MMIO_NODE_virtual_address(node),
                SEP_MMIO_NODE_size(node));
#if __FreeBSD_version >= 1400000
            pmap_unmapdev((void *)(UIOP)SEP_MMIO_NODE_map_token(node), SEP_MMIO_NODE_size(node));
#else
            pmap_unmapdev((vm_offset_t)(UIOP)SEP_MMIO_NODE_map_token(node), SEP_MMIO_NODE_size(node));
#endif
            SEP_MMIO_NODE_size(node)             = 0;
            SEP_MMIO_NODE_map_token(node)        = 0;
            SEP_MMIO_NODE_virtual_address(node)  = 0;
            SEP_MMIO_NODE_physical_address(node) = 0;
        }
        else {
            SEP_DRV_LOG_TRACE("Already unmapped.");
        }
    }

    SEP_DRV_LOG_INIT_OUT("Unmapped node %p.", node);
    return;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn U32 PCI_MMIO_Read_U32(virtual_address_base, offset)
 *
 * @param   virtual_address_base   - Virtual address base
 * @param   offset                 - Register offset
 *
 * @return  U32 read from an MMIO register
 *
 * @brief   Reads U32 value from MMIO
 *
 */
extern U32
PCI_MMIO_Read_U32 (
    U64   virtual_address_base,
    U32   offset
)
{
    U32  temp_u32 = 0LL;
    U32 *computed_address;

    computed_address = (U32*)(((char*)(UIOP)virtual_address_base) + offset);

    SEP_DRV_LOG_REGISTER_IN("Will read U32(0x%llx + 0x%x = 0x%p).", virtual_address_base, offset, computed_address);

    if (!virtual_address_base) {
        SEP_DRV_LOG_ERROR("Invalid base for U32(0x%llx + 0x%x = 0x%p)!", virtual_address_base, offset, computed_address);
        temp_u32 = 0;
    }
    else {
        temp_u32 = *computed_address;
    }

    SEP_DRV_LOG_REGISTER_OUT("Has read U32(0x%llx + 0x%x): 0x%x.", virtual_address_base, offset, temp_u32);
    return temp_u32;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn U64 PCI_MMIO_Read_U64(virtual_address_base, offset)
 *
 * @param   virtual_address_base   - Virtual address base
 * @param   offset                 - Register offset
 *
 * @return  U64 read from an MMIO register
 *
 * @brief   Reads U64 value from MMIO
 *
 */
extern U64
PCI_MMIO_Read_U64 (
    U64   virtual_address_base,
    U32   offset
)
{
    U64  temp_u64 = 0LL;
    U64 *computed_address;

    computed_address = (U64*)(((char*)(UIOP)virtual_address_base) + offset);

    SEP_DRV_LOG_REGISTER_IN("Will read U64(0x%llx + 0x%x = 0x%p).", virtual_address_base, offset, computed_address);

    if (!virtual_address_base) {
        SEP_DRV_LOG_ERROR("Invalid base for U32(0x%llx + 0x%x = 0x%p)!", virtual_address_base, offset, computed_address);
        temp_u64 = 0;
    }
    else {
        temp_u64 = *computed_address;
    }

    SEP_DRV_LOG_REGISTER_OUT("Has read U64(0x%llx + 0x%x): 0x%llx.", virtual_address_base, offset, temp_u64);
    return temp_u64;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void PCI_MMIO_Write_U32(virtual_address_base, offset, value)
 *
 * @param   virtual_address_base   - Virtual address base
 * @param   offset                 - Register offset
 * @param   value                  - Value to write
 *
 * @return  U32 write into an MMIO register
 *
 * @brief   Writes U32 value to MMIO
 *
 */
extern void
PCI_MMIO_Write_U32 (
    U64   virtual_address_base,
    U32   offset,
    U32   value
)
{
    U32 *computed_address;

    computed_address = (U32*)(((char*)(UIOP)virtual_address_base) + offset);

    SEP_DRV_LOG_REGISTER_IN("Writing 0x%x to U32(0x%llx + 0x%x = 0x%p).", value, virtual_address_base, offset, computed_address);

    if (!virtual_address_base) {
        SEP_DRV_LOG_ERROR("Invalid base for U32(0x%llx + 0x%x = 0x%p)!", virtual_address_base, offset, computed_address);
    }
    else {
        *computed_address = value;
    }

    SEP_DRV_LOG_REGISTER_OUT("Has written 0x%x to U32(0x%llx + 0x%x).", value, virtual_address_base, offset);
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn void PCI_MMIO_Write_U64(virtual_address_base, offset, value)
 *
 * @param   virtual_address_base   - Virtual address base
 * @param   offset                 - Register offset
 * @param   value                  - Value to write
 *
 * @return  U64 write into an MMIO register
 *
 * @brief   Writes U64 value to MMIO
 *
 */
extern void
PCI_MMIO_Write_U64 (
    U64   virtual_address_base,
    U32   offset,
    U64   value
)
{
    U64 *computed_address;

    computed_address = (U64*)(((char*)(UIOP)virtual_address_base) + offset);

    SEP_DRV_LOG_REGISTER_IN("Writing 0x%llx to U64(0x%llx + 0x%x = 0x%p).", value, virtual_address_base, offset, computed_address);

    if (!virtual_address_base) {
        SEP_DRV_LOG_ERROR("Invalid base for U32(0x%llx + 0x%x = 0x%p)!", virtual_address_base, offset, computed_address);
    }
    else {
        *computed_address = value;
    }

    SEP_DRV_LOG_REGISTER_OUT("Has written 0x%llx to U64(0x%llx + 0x%x).", value, virtual_address_base, offset);
    return;
}
