/*
 * Copyright (C) 2009 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

#include "lwpmudrv_defines.h"
#include "lwpmudrv_types.h"
#include "lwpmudrv.h"
#include "lwpmudrv_ioctl.h"

#include "control.h"
#include "pax.h"
#include "pax_shared.h"

#include <sys/malloc.h>
#include <sys/kernel.h>
#include <sys/module.h>
#include <sys/types.h>
#include <sys/ioccom.h>
#include <sys/conf.h>
#include <sys/param.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <sys/sx.h>

MALLOC_DEFINE(M_PAX, "pax", "pax driver memory allocations");

#ifndef P_KILLED
#define P_KILLED(x)	0
#endif

static int pax_modevent(module_t mod, int type, void *arg)
{
    int error = 0;

    switch (type) {
    case MOD_LOAD:
	error = pax_Load();
	break;
    case MOD_UNLOAD:
	pax_Unload();
	break;
    default:
	break;
    }

    return (error);
}

moduledata_t pax_mod = {
	"pax",
	(modeventhand_t)pax_modevent,
	0
};

DECLARE_MODULE(pax, pax_mod, SI_SUB_SMP, SI_ORDER_ANY);
MODULE_VERSION(pax, 0);

typedef struct PAX_DEV_NODE_S  PAX_DEV_NODE;
typedef        PAX_DEV_NODE   *PAX_DEV;

struct PAX_DEV_NODE_S {
  long              buffer;
  struct cdev       *cdev;
};

#define PAX_DEV_buffer(dev)   (dev)->buffer
#define PAX_DEV_cdev(dev)     (dev)->cdev

// global variables for the PAX driver

PAX_DEV                 pax_control  = NULL;   // main control
static PAX_VERSION_NODE pax_version;           // version of PAX
static PAX_INFO_NODE    pax_info;              // information on PAX
static PAX_STATUS_NODE  pax_status;            // PAX reservation status

#if defined (DRV_ANDROID)
static struct class     *pax_class   = NULL;
#endif

// Print macros for kernel debugging

//#define DEBUG
#if defined(DEBUG)
#define PAX_PRINT_DEBUG(fmt,args...) { printf("PAX: [DEBUG] " fmt,##args); }
#else
#define PAX_PRINT_DEBUG(fmt,args...) {;}
#endif
#define PAX_PRINT(fmt,args...) { printf("PAX: " fmt,##args); }
#define PAX_PRINT_WARNING(fmt,args...) { printf("PAX: [Warning] " fmt,##args); }
#define PAX_PRINT_ERROR(fmt,args...) { printf("PAX: [ERROR] " fmt,##args); }

/* ------------------------------------------------------------------------- */
/*!
 * @fn     void pax_Init()
 *
 * @param  none
 *
 * @return none
 *
 * @brief  Initialize PAX system
 *
 * <I>Special Notes</I>
 */
static void
pax_Init (
    VOID
)
{
    //
    // Initialize PAX driver version (done once at driver load time)
    //
    PAX_VERSION_NODE_major(&pax_version)  = PAX_MAJOR_VERSION;
    PAX_VERSION_NODE_minor(&pax_version)  = PAX_MINOR_VERSION;
    PAX_VERSION_NODE_bugfix(&pax_version) = PAX_BUGFIX_VERSION;

    // initialize PAX_Info
    pax_info.version       = PAX_VERSION_NODE_version(&pax_version);
    pax_info.managed_by    = 1;   // THIS_MODULE->name;

    // initialize PAX_Status
    pax_status.guid        = PAX_GUID_UNINITIALIZED;
    pax_status.pid         = 0;
    pax_status.start_time  = 0;
    pax_status.is_reserved = PAX_PMU_UNRESERVED;

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn     void pax_Cleanup()
 *
 * @param  none
 *
 * @return none
 *
 * @brief  UnInitialize PAX system
 *
 * <I>Special Notes</I>
 */
static void
pax_Cleanup (
    VOID
)
{
    // uninitialize PAX_Info
    pax_info.managed_by = 0;

    // uninitialize PAX_Status
    pax_status.guid        = PAX_GUID_UNINITIALIZED;
    pax_status.pid         = 0;
    pax_status.start_time  = 0;
    pax_status.is_reserved = PAX_PMU_UNRESERVED;

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn     U32 pax_Process_Valid()
 *
 * @param  U32 pid - process ID
 *
 * @return TRUE or FALSE
 *
 * @brief  Check whether process with pid still exists, and if so,
 *         whether it is still "alive".  If so, then process is
 *         deemed valid.  Otherwise, process is deemed invalid.
 *
 * <I>Special Notes</I>
 */
static U32
pax_Process_Valid (
    U32 pid
)
{
    struct proc       *process;
    U32                valid_process;

    process = pfind(pax_status.pid);
    if ( (process == NULL) ||
         (process->p_state == PRS_ZOMBIE ||
         P_SHOULDSTOP(process) ||
         P_KILLED(process)) )
    {
        // not a valid process
        valid_process = FALSE;
    } else {
        // process is "alive", so assume it is still valid ...
        valid_process = TRUE;
    }

    if (process) {
        PROC_UNLOCK(process);
    }

    return valid_process;
}

// **************************************************************************
//
// below are PAX Open/Read/Write device functions (appears in /proc/kallsyms)
//
// **************************************************************************

// **************************************************************************
//
// below are PAX IOCTL function handlers
//
// **************************************************************************

/* ------------------------------------------------------------------------- */
/*!
 * @fn     OS_STATUS pax_Get_Info()
 *
 * @param  IOCTL_ARGS arg  - pointer to the output buffer
 *
 * @return OS_STATUS
 *
 * @brief  Local function that handles the PAX_IOCTL_INFO call
 *         Returns static information related to PAX (e.g., version)
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
pax_Get_Info (
    IOCTL_ARGS   arg
)
{
    int error;

    if ((error=copy_to_user((PAX_INFO_NODE *)(arg->buf_usr_to_drv), &pax_info, sizeof(PAX_INFO_NODE)))) {
        PAX_PRINT_ERROR("pax_Get_Info: unable to copy to user (error=%d)!\n", error);
        return OS_FAULT;
    }

    PAX_PRINT_DEBUG("pax_Get_Info: sending PAX info (%ld bytes):\n", sizeof(PAX_INFO_NODE));
    PAX_PRINT_DEBUG("pax_Get_Info:      raw_version = %u (0x%x)\n", pax_info.version, pax_info.version);
    PAX_PRINT_DEBUG("pax_Get_Info:            major = %u\n", PAX_VERSION_NODE_major(&pax_version));
    PAX_PRINT_DEBUG("pax_Get_Info:            minor = %u\n", PAX_VERSION_NODE_minor(&pax_version));
    PAX_PRINT_DEBUG("pax_Get_Info:           bugfix = %u\n", PAX_VERSION_NODE_bugfix(&pax_version));
    PAX_PRINT_DEBUG("pax_Get_Info:      managed_by = 0x%lu\n", (long unsigned int)pax_info.managed_by);
    PAX_PRINT_DEBUG("pax_Get_Info: information sent.\n");

    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn     OS_STATUS pax_Get_Status()
 *
 * @param  IOCTL_ARGS arg  - pointer to the output buffer
 *
 * @return OS_STATUS
 *
 * @brief  Local function that handles the PAX_IOCTL_STATUS call
 *         Returns status of the reservation (e.g., who owns)
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
pax_Get_Status (
    IOCTL_ARGS   arg
)
{
    int error;

    if ((error=copy_to_user((PAX_STATUS_NODE *)(arg->buf_usr_to_drv), &pax_status, sizeof(PAX_STATUS_NODE)))) {
        PAX_PRINT_ERROR("pax_Get_Status: unable to copy to user (error=%d)!\n", error);
        return OS_FAULT;
    }

    PAX_PRINT_DEBUG("pax_Get_Status: sending PAX status (%ld bytes):\n", sizeof(PAX_STATUS_NODE));
    PAX_PRINT_DEBUG("pax_Get_Status:    guid = 0x%lu\n", (long unsigned int)pax_status.guid);
    PAX_PRINT_DEBUG("pax_Get_Status:    pid = %lu\n", (long unsigned int)pax_status.pid);
    PAX_PRINT_DEBUG("pax_Get_Status:    start_time = %lu\n", (long unsigned int)pax_status.start_time);
    PAX_PRINT_DEBUG("pax_Get_Status:    is_reserved = %u\n", pax_status.is_reserved);
    PAX_PRINT_DEBUG("pax_Get_Status: status sent.\n");

    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn     OS_STATUS pax_Unreserve()
 *
 * @param  none
 *
 * @return OS_STATUS
 *
 * @brief  Local function that handles the PAX_IOCTL_UNRESERVE call
 *         Returns OS_SUCCESS if PMU unreservation succeeded, otherwise failure
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
pax_Unreserve (
    VOID
)
{
    // if no reservation is currently held, then return success
    if (pax_status.is_reserved == PAX_PMU_UNRESERVED) {
        PAX_PRINT_DEBUG("pax_Unreserve: currently unreserved\n");
        return OS_SUCCESS;
    }

    // otherwise, there is a reservation ...
    // allow the process which started the reservation to unreserve
    // or if that process is invalid, then any other process can unreserve
    if ( (pax_status.pid == curproc->p_pid) ||
         (! pax_Process_Valid(pax_status.pid)) ) {
        S32 reservation = -1;
        PAX_PRINT_DEBUG("pax_Unreserve: pid %d attempting to unreserve PMU held by pid %lld\n", curproc->p_pid, pax_status.pid);
        reservation = atomic_cmpset_int(&pax_status.is_reserved, PAX_PMU_RESERVED, PAX_PMU_UNRESERVED);
        if (reservation < 0) {
        // no-op ... eliminates "variable not used" compiler warning
        }
        PAX_PRINT_DEBUG("pax_Unreserve: reserve=%d, is_reserved=%d\n", reservation, pax_status.is_reserved);
        // unreserve but keep track of last PID/GUID that had reservation
    }

    PAX_PRINT_DEBUG("pax_Unreserve: pid %d unreserve status: %d\n", curproc->p_pid, pax_status.is_reserved);

    return ((pax_status.is_reserved == PAX_PMU_UNRESERVED) ? OS_SUCCESS : OS_FAULT);
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn     OS_STATUS pax_Reserve_All()
 *
 * @param  none
 *
 * @return OS_STATUS
 *
 * @brief  Local function that handles the PAX_IOCTL_RESERVE_ALL call
 *         Returns OS_SUCCESS if PMU reservation succeeded, otherwise failure
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
pax_Reserve_All (
    VOID
)
{
    S32 success;

    // check if PMU can be unreserved
    if (pax_status.is_reserved == PAX_PMU_RESERVED) {
        OS_STATUS unreserve_err = pax_Unreserve();
        if (unreserve_err != OS_SUCCESS) {
            return unreserve_err; // attempt to unreserve failed, so return error
        }
    }

    PAX_PRINT_DEBUG("pax_Reserve_All: pid %d attempting to reserve PMU\n",curproc->p_pid);

    // at this point, there is no reservation, so commence race to reserve ...
    success = atomic_cmpset_int(&pax_status.is_reserved, PAX_PMU_UNRESERVED, PAX_PMU_RESERVED);

    // only one request to reserve will succeed, and when it does, update status
    // information with the successful request
    if (success)
    {
      /*
        // TODO: define a GUID that uniquely identifies the session
        ts.tv_sec = current->start_time.tv_sec;
        ts.tv_nsec = current->start_time.tv_nsec;
        pax_status.guid = (U64)timespec_to_ns(&ts);
        // NOTE: the following "wallclock" kernel functions/structures are not exported ...
           #include <linux/time.h>
           struct timespec ts;
           struct timesval tv;
           do_gettimeofday(&tv)
           getnstimeofday(&ts)
           current_kernel_time()
           (U64)timespec_to_ns(&ts)
      */
        pax_status.start_time = rdtsc();
        pax_status.pid = curproc->p_pid;

        return OS_SUCCESS;
    }

    return OS_FAULT;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn     OS_STATUS pax_Reserve_All()
 *
 * @param  inode - pointer to the device object
 * @param  filp  - pointer to the file object
 * @param  cmd   - ioctl value (defined in lwpmu_ioctl.h)
 * @param  arg   - arg or arg pointer
 *
 * @return OS_STATUS
 *
 * @brief  Worker function that handles IOCTL requests from the user mode
 *
 * <I>Special Notes</I>
 */
static int
pax_Device_Control (
    struct cdev		*cdev,
    u_long		 cmd,
    caddr_t		 arg,
    int			 flag,
    struct thread  	*td
)
{
    int             status = OS_SUCCESS;
    IOCTL_ARGS_NODE local_args;

    memcpy(&local_args, (IOCTL_ARGS)arg, sizeof(local_args));

    PAX_PRINT_DEBUG("device control (0x%lx) called\n", cmd);

    // dispatch to appropriate PAX IOCTL function
    switch (cmd) {
        case PAX_IOCTL_INFO:
            PAX_PRINT_DEBUG("PAX_IOCTL_INFO\n");
            status = pax_Get_Info(&local_args);
            break;

        case PAX_IOCTL_STATUS:
            PAX_PRINT_DEBUG("PAX_IOCTL_STATUS\n");
            status = pax_Get_Status(&local_args);
            break;

        case PAX_IOCTL_RESERVE_ALL:
            PAX_PRINT_DEBUG("PAX_IOCTL_RESERVE_ALL\n");
            status = pax_Reserve_All();
            break;

        case PAX_IOCTL_UNRESERVE:
            PAX_PRINT_DEBUG("PAX_IOCTL_UNRESERVE\n");
            status = pax_Unreserve();
            break;

        default:
            PAX_PRINT_ERROR("unknown IOCTL group:%ld basecmd:%ld\n",
                            IOCGROUP(cmd), IOCBASECMD(cmd));
            status = OS_ILLEGAL_IOCTL;
            break;
    }

    PAX_PRINT_DEBUG("cmd: %ld group:%ld basecmd:%ld, returned %d\n",
                    cmd, IOCGROUP(cmd), IOCBASECMD(cmd), status);

    return status;
}

// **************************************************************************
//
// PAX device file operation definitions (required by kernel)
//
// **************************************************************************

/*
 * Structure that declares the usual file access functions
 * First one is for pax, the control functions
 */
static struct cdevsw pax_Fops = {
    .d_version =	D_VERSION,
    .d_flags =		0,
    .d_ioctl =		pax_Device_Control
};

/* ------------------------------------------------------------------------- */
/*!
 * @fn     int pax_Setup_Cdev()
 *
 * @param  dev    - pointer to the device object
 * @param  devnum - major/minor device number
 * @param  fops   - point to file operations struct
 *
 * @return int
 *
 * @brief  Set up functions to be handled by PAX device
 *
 * <I>Special Notes</I>
 */
static void
pax_Setup_Cdev (
    PAX_DEV                 dev,
    struct cdevsw          *cdevsw,
    char *		    dev_name
)
{
    int gid = GID_OPERATOR;
    int perm = 0660;        // default permission (requires prefix 0 for octal format)

    TUNABLE_INT_FETCH("hw.pax.gid", &gid);
    TUNABLE_INT_FETCH("hw.pax.perm", &perm);

    PAX_DEV_cdev(dev) = make_dev(cdevsw, 0, UID_ROOT, gid, perm, "%s", dev_name);
}

// **************************************************************************
//
// Exported PAX functions (see pax.h) ; will appear under /proc/kallsyms
//
// **************************************************************************

/* ------------------------------------------------------------------------- */
/*!
 * @fn     int pax_Load()
 *
 * @param  none
 *
 * @return int
 *
 * @brief  Load the PAX subsystem
 *
 * <I>Special Notes</I>
 */
extern int
pax_Load (
    VOID
)
{
    int result = 0;

    PAX_PRINT_DEBUG("checking for %s interface...\n", PAX_NAME);

    /* If PAX interface does not exist, create it */

    /* Allocate memory for the PAX control device */
    pax_control = malloc(sizeof(PAX_DEV_NODE), M_PAX, M_NOWAIT);
    if (! pax_control) {
        PAX_PRINT_ERROR("Unable to allocate memory for %s device\n", PAX_NAME);
        return OS_NO_MEM;
    }
    /* Initialize memory for the PAX control device */
    memset(pax_control, '\0', sizeof(PAX_DEV_NODE));
    /* Register PAX file operations with the OS */
    pax_Setup_Cdev(pax_control,&pax_Fops,PAX_NAME);

    pax_Init();

    //
    // Display driver version information
    //
    PAX_PRINT("PMU arbitration service v%d.%d.%d has been started.\n",
              PAX_VERSION_NODE_major(&pax_version),
              PAX_VERSION_NODE_minor(&pax_version),
              PAX_VERSION_NODE_bugfix(&pax_version));

    return result;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn     int pax_Unload()
 *
 * @param  none
 *
 * @return none
 *
 * @brief  Unload the PAX subsystem
 *
 * <I>Special Notes</I>
 */
extern VOID
pax_Unload (
    VOID
)
{
    // warn if unable to unreserve
    if (pax_Unreserve() != OS_SUCCESS) {
        PAX_PRINT_WARNING("Unloading driver with existing reservation ....");
        PAX_PRINT_WARNING("         guid = 0x%lu\n", (long unsigned int)pax_status.guid);
        PAX_PRINT_WARNING("          pid = %ld\n", (long int)pax_status.pid);
        PAX_PRINT_WARNING("   start_time = %lu\n", (long unsigned int)pax_status.start_time);
        PAX_PRINT_WARNING("  is_reserved = %u\n", pax_status.is_reserved);
    }

    if (pax_control != NULL) {
        destroy_dev(PAX_DEV_cdev(pax_control));
        free(pax_control, M_PAX);
    }

    //
    // Display driver version information
    //
    PAX_PRINT("PMU arbitration service v%d.%d.%d has been stopped.\n",
              PAX_VERSION_NODE_major(&pax_version),
              PAX_VERSION_NODE_minor(&pax_version),
              PAX_VERSION_NODE_bugfix(&pax_version));

    // clean up resources used by PAX
    pax_Cleanup();

    return;
}

