/*
 * Copyright (C) 2005 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

#include <sys/unistd.h>
#include <sys/kthread.h>
#include <sys/param.h>
#include <sys/conf.h>
#include <sys/ioccom.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/module.h>
#include <sys/pmc.h>
#include <sys/pmckern.h>
#include <sys/smp.h>
#include <sys/systm.h>
#include <sys/uio.h>
#ifdef DRV_USE_NMI
#include <sys/bus.h>
#include <sys/pmckern.h>

#include <machine/intr_machdep.h>
#if __FreeBSD_version >= 1100006
#include <x86/apicvar.h>
#else
#include <machine/apicvar.h>
#endif
#endif

#include "lwpmudrv_defines.h"
#include "lwpmudrv_version.h"
#include "lwpmudrv_types.h"
#include "lwpmudrv_ecb.h"
#include "lwpmudrv_struct.h"
#include "lwpmudrv.h"
#include "utility.h"
#include "control.h"
#include "output.h"
#include "pmi.h"

/* ------------------------------------------------------------------------- */
/*!
 *  @fn       VOID sep_capture_user_callchains (struct thread *td, struct trapframe *tf)
 *
 *  @brief    capture user space call chain records
 *
 *  @param    defined by FreeBSD PMC subsystem
 *  @return   None
 */ 
static void
sep_capture_user_callchains(
    struct thread    *td,
    struct trapframe *tf
)
{
    U32		ccsize, i;
    uintptr_t	*ccbuffer;
    CPU_STATE	pcpu;
    int         cpu = curcpu;
    pcpu     = &pcb[cpu];

    SEP_DRV_LOG_TRACE_IN("");

    /*
     * Look through the CPU_STATE size array, looking for any
     *  non-zero values.  If we find any, capture a user callchain
     *  at the associated buffer address.
     *
     * A single NMI may be due to multiple counter overflows, so we could
     *  need to save the user callchain in multiple places - hence the
     *  need to look through the entire size array and not just the first
     *  non-zero entry.
     */
    for (i = 0; i < MAX_CC_PER_INTERRUPT; ++i) {
        ccsize = CPU_STATE_user_callchain_size(pcpu, i);
	ccbuffer = CPU_STATE_user_callchain_buffer(pcpu, i);

	if (ccsize > 0) {
	    sep_save_user_callchain(ccbuffer, ccsize, tf);

     	    /*
  	     * Set size back to 0, effectively clearing this
 	     *  entry until another user callchain notification
	     *  needs to be saved here.
   	     */
	    CPU_STATE_user_callchain_size(pcpu, i) = 0;
	}
    }

    /*
     * One of the sample buffers overflowed during the PMI handler, but
     *  we deferred setting the pmc_cpumask until after the user
     *  callchain was populated.
     */
    if (CPU_STATE_deferred_reap_samples(pcpu)) {
#if __FreeBSD_version >= 1200067
	DPCPU_SET(pmc_sampled, 1);
#elif __FreeBSD_version >= 900040
	CPU_SET_ATOMIC(cpu, &pmc_cpumask);
#else
	atomic_set_rel_int(&pmc_cpumask, (1 << cpu));
#endif
	CPU_STATE_deferred_reap_samples(pcpu) = 0;
    }

    td->td_pflags &= ~TDP_CALLCHAIN;
    sched_unpin();

    SEP_DRV_LOG_TRACE_OUT("");
}
/* ------------------------------------------------------------------------- */
/*!
 *  @fn       int sep_hook_handler(struct thread *td, int function, void *arg)
 *
 *  @brief    This handler hooks into the FreeBSD kernel's PMC subsystem.  Currently,
 *            we use it only for signaling sample collector threads when buffers are
 *            full.
 *
 *  @param    defined by FreeBSD PMC subsystem 
 *  @return   0
 * 
 *  <I>Special Notes:</I>
 *  TODO: Use this to handle shared library unmaps, since the the event
 *  registration mechanism for process exit doesn't work for unmaps.
 */
int
sep_hook_handler (
    struct thread *td,
    int            function,
    void          *arg
)
{
    SEP_DRV_LOG_NOTIFICATION_IN("Start of call with function: %d.", function);
    int cpu;

    switch (function)
    {
    case PMC_FN_USER_CALLCHAIN:
	sep_capture_user_callchains(td, (struct trapframe *)arg);
	break;

    case PMC_FN_PROCESS_EXEC:
    case PMC_FN_CSW_IN:
    case PMC_FN_CSW_OUT:
#ifdef PMC_FN_KLD_LOAD
    case PMC_FN_KLD_LOAD:
    case PMC_FN_KLD_UNLOAD:
#endif
    case PMC_FN_MMAP:
    case PMC_FN_MUNMAP:
#ifdef PMC_FN_USER_CALLCHAIN_SOFT
    case PMC_FN_USER_CALLCHAIN_SOFT:
#endif
#ifdef PMC_FN_SOFT_SAMPLING
    case PMC_FN_SOFT_SAMPLING:
#endif
	break;
    case PMC_FN_DO_SAMPLES:
        SEP_DRV_LOG_TRACE("Start of PMC_FN_DO_SAMPLES.");
        cpu = curcpu;
#if __FreeBSD_version >= 1200067
	DPCPU_SET(pmc_sampled, 0);
#elif __FreeBSD_version >= 900040
	CPU_CLR_ATOMIC(cpu, &pmc_cpumask);
#else
	atomic_clear_int(&pmc_cpumask, (1 << cpu));
#endif
	cv_signal(&BUFFER_DESC_cv(&cpu_buf[cpu]));
        SEP_DRV_LOG_TRACE("End of of PMC_FN_DO_SAMPLES.");
	break;

     default:
	SEP_DRV_LOG_ERROR("Unknown PMC hook %d.", function);
	break;
     }

    SEP_DRV_LOG_NOTIFICATION_OUT("End of call with function: %d.", function);
    return 0;
}
/*
 * pmc_intr and pmc_hook are the key callback interfaces
 * exported from FreeBSD pmc subsystem which defining 
 * default behaviors 
 * 
 * refer to pmc_initialize() in /sys/dev/hwpmc/hwpmc_mod.c
 * pmc_intr - handling low level PMU related interrupt register only
 * pmc_hook - portal or global entry to handle sampling records and
 *            sync up with rest of FreeBSD kernel subsystems 
 *
 */
/* ------------------------------------------------------------------------- */
/*!
 *  @fn       VOID FBSD_PMC_Register_PMI_Handler 
 *
 *  @brief    register PMI handler and global entry into FreeBSD PMC
 *
 *  @param    None
 *  @return   None
 */ 
void 
FBSD_PMC_Register_PMI_Handler(void) 
{ 
    pmc_intr = _PMI_Interrupt_Handler;
    pmc_hook = sep_hook_handler;
}
/* ------------------------------------------------------------------------- */
/*!
 *  @fn       VOID FBSD_PMC_Unregister_PMI_Handler 
 *
 *  @brief    register PMI handler and global entry from FreeBSD PMC
 *
 *  @param    None
 *  @return   None
 */ 
void 
FBSD_PMC_Unregister_PMI_Handler(void) 
{ 
    pmc_intr = NULL;
    pmc_hook = NULL;
}
