# (c) 1992-2024 Intel Corporation.                                              
# Intel, the Intel logo, Intel, MegaCore, NIOS II, Quartus and TalkBack         
# words and logos are trademarks of Intel Corporation or its                    
# subsidiaries in the U.S. and/or other countries. Other marks and              
# brands may be claimed as the property of others.                              
# See Trademarks on intel.com for full list of Intel trademarks or the          
# Trademarks & Brands Names Database (if Intel)                                 
# or See www.Intel.com/legal (if Altera)                                        
# Your use of Intel Corporation's design tools, logic functions and             
# other software and tools, and its AMPP partner logic functions, and           
# any output files any of the foregoing (including device programming           
# or simulation files), and any associated documentation or information         
# are expressly subject to the terms and conditions of the Altera               
# Program License Subscription Agreement, Intel MegaCore Function               
# License Agreement, or other applicable license agreement, including,          
# without limitation, that your use is for the sole purpose of                  
# programming logic devices manufactured by Intel and sold by Intel or          
# its authorized distributors.                                                  
# Please refer to the applicable agreement for further details.                 


#  Inputs:  A target
#           A source file
#           An optional object file name
#  Output:  An object file sutable for the target platform
#
#
# Example:
#     Command:
#        fpga_crossgen -o prim.o prim.xml --cpp_model prim.cpp
#     Generates:
#        prim.o - object file suitable for inclusion into libraries
#
# vim: set ts=2 sw=2 et

BEGIN {
  unshift @INC, (
    grep { -d $_ } (
      map { $ENV{INTELFPGAOCLSDKROOT} . $_ }
        qw(
        /host/windows64/bin/perl/lib/MSWin32-x64-multi-thread
        /host/windows64/bin/perl/lib
        /share/lib/perl
        /share/lib/perl/5.8.8 )
    )
  );
}


package FPGA_Crossgen;
use strict;
use Exporter;
require acl::Env;
require acl::Common;
require acl::File;
require acl::Pkg;
require acl::HLS_Pkg;
use acl::AOCDriverCommon;
use Cwd;

# Executables
my $llc_exe = $ENV{INTELFPGAOCLSDKROOT} . '/llvm/aocl-bin'."/aocl-llc";
my $prog = acl::File::mybasename($0);
$prog =~ s/.pl$//;

my $usage =<<USAGE;
Usage: $prog -o <lib>.o <file>.xml [opt]
Options:
    -h, --help         Print this help, then exit successfully.
    -n, --dry-run      Don\'t actually do anything, just check command line
    --cpp_model <file>.cpp
                       Add cpp model. Only valid if main file is xml
    -o <object name>   Make the new object file be named <object name>
    [opt]              Anything not recognized will be sent to the
                       underlying compiler
USAGE

# checks host OS, returns true for linux, false for windows.
sub isLinux {
  if ($^O eq 'linux') {
    return 1;
  }
  return;
}

# Prefered to die, since die always prints where we bailed out (line number)
sub myexit {
  my $exit_code = shift;
  if (scalar @_){
    my $message = shift;
    print STDERR "Error: $message\n";
  }
  exit $exit_code;
}

sub mybail {
  myexit (1, @_);
}

my $SYCL_AOCO_SECTION = 'sycl-fpga_aoco-intel-unknown';
my $SYCL_X86_SECTION = isLinux() ? 'host-x86_64-unknown-linux-gnu' : 'host-x86_64-pc-windows-msvc';
my $AOC_X86_SECTION = isLinux() ? '.acl.clang_ir.x86_64-unknown-linux-intelfpga' : '.acl.clang_ir.x86_64-pc-windows-intelfpga';

sub create_sycl_library_object {
  my ($source_file, $object_file, $object_suffix, $cpp_model, $compiler_args, $toolchain, $verbose) = @_;
  print "Creating SYCL Library object from $source_file for $object_file\n" if $verbose;

  my $tmp_aoco_file = $object_file;
  my $tmp_x86_bc_file = $object_file;
  my $tmp_x86_obj_file = $object_file;

  $tmp_aoco_file =~ s/\.$object_suffix/\.tmp\.aoco/;
  $tmp_x86_bc_file =~ s/\.$object_suffix/_x86\.tmp\.bc/;
  $tmp_x86_obj_file =~ s/\.$object_suffix/_x86\.tmp\.$object_suffix/;

  # Help with using RTL emulation model in SYCL
  if(length $toolchain){
    push @$compiler_args, ('-preproc-gcc-toolchain', $toolchain);
  }
  # Only supports SYCL emulation model in RTL libraries
  # Not the flow `fpga_crossgen --target ocl --source sycl`
  print "Packaging RTL described by $source_file into OpenCL Library object $tmp_aoco_file for SYCL emulation model\n" if $verbose;
  # note that here "aocl xlibrary" seems to basically just be a wrapper to a call to aocl-libedit
  (acl::Common::mysystem_full({}, "aocl xlibrary createobject $source_file -o $tmp_aoco_file")==0) or myexit ($?>>8);
  print "Creating Emulation model from $source_file and packaging it into OpenCL Library object $object_file\n" if $verbose;

  # Extract x86 section from .aoco then put x86 part and .aoco in sections of SYCL object
  if(!$cpp_model) {
    # No emulation model means no x86 section in the .aoco
    # But SYCL still requires one so make it empty
    (acl::Common::mysystem_full({}, "touch $tmp_x86_bc_file")==0) or myexit ($?>>8);
    (acl::Common::mysystem_full({}, "$llc_exe -filetype=obj -o $tmp_x86_obj_file $tmp_x86_bc_file")==0) or myexit ($?>>8);
  } else {
    (acl::Common::mysystem_full({}, "icpx -fsycl -fintelfpga -c $cpp_model -o $tmp_x86_obj_file @$compiler_args $toolchain")==0) or myexit ($?>>8);
  }
  my $offload_bundler = qx{icpx --print-prog-name=clang-offload-bundler};
  chomp($offload_bundler);
  (acl::Common::mysystem_full({}, "$offload_bundler -type=o -targets=$SYCL_X86_SECTION,$SYCL_AOCO_SECTION -input=$tmp_x86_obj_file -input=$tmp_aoco_file -output=$object_file")==0) or myexit ($?>>8);

  # Clean up temp files
  acl::File::remove_tree($tmp_aoco_file);
  acl::File::remove_tree($tmp_x86_bc_file);
  acl::File::remove_tree($tmp_x86_obj_file);
}

sub run() {
  my $source_file = undef;
  my $object_file = undef;
  my $object_suffix = undef;
  my $cpp_model = undef;
  my @compiler_args;
  my $verbose = undef;
  my $dry_run = 0;
  my $sycl_rtl_lib = 0;

  my $toolchain_dir= $ENV{INTELFPGAOCLSDKROOT} . '/linux64/bin'."/../../../gcc";
  # Toolchain priority:
  # 1. OVERRIDE:  Command line argument
  # 2. STANDARD:  Installed directory
  # 3. FALL-BACK: Clang analysis
  #
  # Make "directory" override "analysis"
  # (this variable can still be overwritten by "argument" parsing)
  my $toolchain = (isLinux() && -d $toolchain_dir) ? $toolchain_dir : undef;

  while (@ARGV) {
    my $arg = shift @ARGV;
    if ( ($arg eq '-n') || ($arg eq '--dry-run') ) {
        $dry_run = 1;
    } elsif ( ($arg eq '-h') || ($arg eq '--help') ) {
        print $usage; exit 0;
    } elsif ($arg eq '-v') {
        $verbose +=1;

    } elsif ( $arg eq '--cpp_model' ) {
      ($#ARGV >= 0 and $ARGV[0] !~ m/^-./) or mybail "Option $arg requires a file argument.";
      $arg = shift @ARGV;
      if ($arg =~ m/\.cpp$/) {
        (!$cpp_model) or mybail "Only one C++ model file is allowed\n";
        $cpp_model = $arg;
      } else {
        mybail "C++ model file has to be .cpp\n";
      }
    } elsif ( $arg =~ m/^--gcc-toolchain=/ ) {
      if (acl::Env::is_windows()) {
        print "Warning: --gcc-toolchain argument ignored\n";
      } else {
        $toolchain = $arg;
        $toolchain =~ s/--gcc-toolchain=//;
      }
    } elsif ( $arg eq '-o' ) {
      ($#ARGV >= 0 and $ARGV[0] !~ m/^-./) or mybail "Option $arg requires a file argument.";
      $object_file = shift @ARGV;

    } elsif ($arg =~ m/\.xml$/) {
      (!$source_file) or mybail "Only one source file is allowed\n";
      $source_file = $arg;
    } elsif ( ( $arg eq '--target' )  ) {
      shift @ARGV;
      print("Warning: The --target flag is deprecated and will be ignored! This tool can only be used to target sycl.\n");
    } elsif ( ( $arg eq '--source' ) ) {
      shift @ARGV;
      print("Warning: The --source flag is deprecated and will be ignored! This tool can only be used with RTL source files.\n");
    } else {
      # Unknown arguments forward to underlying compiler
      push @compiler_args, $arg;
    }
  }

  acl::Common::set_verbose($verbose);
  # Peel of the two -v's we used here and apply any remaining ones to aoc/i++
  for my $i (3..$verbose) {
      push @compiler_args, '-v';
  }

  # Check arguments
  ($source_file) or mybail "RTL source is required";

  my $stem = acl::File::mybasename($source_file);
  my $suffix = $stem;
  $suffix =~ s/.*\.//;
  $stem =~ s/\.$suffix//;

  if(!($suffix eq 'xml')) {
    mybail "Unexpected source file extension: " . $suffix . "\n";
  }

  # Set object file name if none given
  if (!$object_file) {
    if (isLinux()) {
      $object_suffix = 'o';
    } else {
      $object_suffix = 'obj';
    }

    $object_file = $stem.'.'.$object_suffix;
  }

  # Verify that object suffix matches target
  $object_suffix = $object_file;
  $object_suffix =~ s/.*\.//;
  if ( isLinux() ) {
    ($object_suffix eq 'o') or mybail "$object_file object file must have the suffix .o on linux\n";
  } else {
    ($object_suffix eq 'obj') or mybail "$object_file object file must have the suffix .obj on windows\n";
  }

  # Stop before compilation
  if ($dry_run) {
    exit (0);
  }

  create_sycl_library_object($source_file, $object_file, $object_suffix, $cpp_model, \@compiler_args, $toolchain, $verbose);
}

run();
