
=pod

=head1 COPYRIGHT

# (c) 1992-2024 Intel Corporation.                                              
# Intel, the Intel logo, Intel, MegaCore, NIOS II, Quartus and TalkBack         
# words and logos are trademarks of Intel Corporation or its                    
# subsidiaries in the U.S. and/or other countries. Other marks and              
# brands may be claimed as the property of others.                              
# See Trademarks on intel.com for full list of Intel trademarks or the          
# Trademarks & Brands Names Database (if Intel)                                 
# or See www.Intel.com/legal (if Altera)                                        
# Your use of Intel Corporation's design tools, logic functions and             
# other software and tools, and its AMPP partner logic functions, and           
# any output files any of the foregoing (including device programming           
# or simulation files), and any associated documentation or information         
# are expressly subject to the terms and conditions of the Altera               
# Program License Subscription Agreement, Intel MegaCore Function               
# License Agreement, or other applicable license agreement, including,          
# without limitation, that your use is for the sole purpose of                  
# programming logic devices manufactured by Intel and sold by Intel or          
# its authorized distributors.                                                  
# Please refer to the applicable agreement for further details.                 


=cut

package acl::Simulator;
require acl::Common;
require acl::Env;
require acl::File;
require acl::AOCDriverCommon;

@ISA        = qw(Exporter);
@EXPORT     = ();
@EXPORT_OK  = qw(
    set_sim_debug
    set_sim_debug_depth
    set_sim_use_qrun_flow
    set_sim_accurate_memory
    set_sim_kernel_clk_frequency
    set_sim_enable_warnings
    set_sim_elab_options
    set_sim_library_to_aoc
    set_sim_library_to_hls
    query_vsim_arch
    initialize_simulation_paths
    get_sim_compile_exe
    get_sim_run_tcl
    get_sim_package
    get_sim_options
    generate_simulation_scripts
    opencl_create_sim_system
    compile_opencl_simulator
    write_sim_repackage_script
    update_sim_debug_depth
);

use strict;


my $module = 'acl::Simulator';

=head1 NAME

acl::Simulator - Simulator utilities

=head1 VERSION

$Header$

=head1 SYNOPSIS


=head1 DESCRIPTION

This module provides utilities for create the OpenCL and HLS simulation framework.

All methods names may optionally be imported, e.g.:

   use acl::Simulator qw( compile_opencl_simulator );

=head1 METHODS

=cut

# Simulation options set by user
my $sim_debug = 0;    # 1 means add elaboration option to output vsim.wlf when running the simulation
                      # 0 means no waveform to speed up running time and reduce diskspace usage (default)
my $sim_debug_depth = undef;  # sets the hierarchy depth to capture into vsim.wlf. Applicable when $sim_debug=1
                              # Special case is undef, it means capture all, set with only -ghdl flag with no depth #
my $sim_use_qrun_flow = undef;
my $sim_accurate_memory = 0;
my $sim_kernel_clk_frequency = 400;  # in MHz
my $sim_enable_warnings = 0;  # 0 means suppress missing ports and parameter warnings
my $sim_user_elab_options = "";

my $qsys_temp_dir = "aoc_msim.temp";     # directory that contains mpsim.qsys. Default value used by less-QSys option
my $qsys_temp_dir_path = undef;          # absolute directory path name to simulation folder
my $cosim_lib = undef;                   # cosim library determined by flow

# Compilation flow used. 3 levels of QSys:
#  1) Whole simulation framework is generated by QSys - existing simulation framework
#  2) Only the simulation board is generated by QSys  - SDK provide default with skip QSys for kernel system
#  3) Only the kernel_system is generated by QSys     - vendor board sim with QSys-less flow, i.e. S10
my $board_qsys = 0;         # 1: whole simulation framework; 0: simulation board only
my $kernel_system_qsys = 0; # 1: not currently supported

# Simulation working directory and script paths. Set by initialize_simulation_paths()
# QSys infrastructure generates $QSYS_SIMDIR to put all the sim scripts and the RTL and files are located in ip/$QSYS_SIMDIR
my $sim_qsys_name = undef;   # QSys project name.
my $sim_dir = undef;         # Store the variable to define $QSYS_SIMDIR
my $sim_script_dir = undef;  # location of the generated compile and run scripts.
                             # With ModelSim: Qsys sim directory == compile directory, but VCSMX is different
my $sim_compile_script_name = undef;  # msim_compile.tcl'
my $fname_runscript = undef;
my $fname_exe_com_script = undef;
my $fname_view_sys_script = undef; # System-specific script to use for viewing the waveforms
my $fname_view_do_script = undef;  # System-agnostic script to configure modelsim

# OpenCL specific package and kernel system instance name
my @sim_package = ('board_spec.xml');   # for adding files and directories to aocx
my $opencl_qsys_name = "mpsim";         # qsys name match with aoc_sim_generate_qsys.tcl
my $kernel_system_inst = "ks";

my $sycl_mode = 0;

sub set_sim_debug($) {
  $sim_debug = shift;
}

sub set_sim_debug_depth($) {
  $sim_debug_depth = shift;
}

sub set_sim_use_qrun_flow($) {
  $sim_use_qrun_flow = shift;
}

sub set_sim_accurate_memory($) {
  $sim_accurate_memory = shift;
}

sub set_sim_kernel_clk_frequency($) {
  $sim_kernel_clk_frequency = shift;
}

sub set_sim_enable_warnings($) {
  $sim_enable_warnings = shift;
}

sub set_sim_elab_options {
  $sim_user_elab_options = shift;
}

# Set the pre-generated input directory use by OpenCL -sim-input-dir flag
sub set_sim_dir_path($) {
  my $sim_path = shift;
  if (!defined($sim_path)) {
    # user did not provide a path, use default aoc_msim.temp
    $sim_path = $qsys_temp_dir;
  }
  $qsys_temp_dir_path = acl::File::abs_path($sim_path);
  acl::Common::mydie("User specified simulation directory ${sim_path} does not exist: $!") if (! (-e $qsys_temp_dir_path));
  $qsys_temp_dir = acl::File::mybasename($qsys_temp_dir_path);
}

sub set_sim_library_to_aoc {
  $cosim_lib = "aoc_cosim_msim";
}

sub set_sim_library_to_hls {
  $cosim_lib = "hls_cosim_msim";
}

# initialize_simulation_paths needs to be called before generate_simulation_scripts()
# The functions sets the qsys project name, sim_dir that's use for QSYS_SIMDIR,
# directory name sim_script_dir for output scripts dir, and compile and run scripts
# the script names and run scripts changes based on platform and simulator vendor
sub initialize_simulation_paths {
  my ($qsys_name, $standard) = @_;
  $sim_qsys_name = $qsys_name;

  if ($standard) {
    $sim_dir = $sim_qsys_name."/simulation";  # Path structure match QSys
  } else {
    $sim_dir = $sim_qsys_name."/sim";  # Path structure match QSys
  }
  # TODO: add vendor initialize code, i.e. VCS support
  $sim_script_dir = $sim_dir;
  $sim_compile_script_name = "msim_compile.tcl";    # Modelsim filename
  $fname_exe_com_script = acl::Env::is_linux() ? 'compile.sh' : 'compile.cmd';
  $fname_exe_com_script = $sim_script_dir.'/'.$fname_exe_com_script;  # put the script in sim for Modelsim
  $fname_runscript = $sim_script_dir.'/msim_run.tcl';
  # accurate memory model will require ddr, mpsim and others folders to be packaged up
  if ($sim_accurate_memory) {
    # currently it must be used with -sim-input-dir and hardcode in directories
    # some basic error checking before continuing
    if (!(-d $qsys_temp_dir_path)) {
      acl::Common::mydie("Accurate memory simulation folder $qsys_temp_dir_path does not exist!\n");
    }
    foreach my $file (acl::File::simple_glob( "$qsys_temp_dir_path/*" )) {
      # do not copy *kernel_system* from input dir because it's from empty kernel system
      next if ($file =~ /kernel_system/);
      push(@sim_package, acl::File::mybasename($file));
    }
  }
}

sub get_sim_compile_exe {
  if ( acl::Env::is_linux() ) {
    return "sh $fname_exe_com_script";
  }
  elsif ( acl::Env::is_windows() ) {
    return "call $fname_exe_com_script";
  }
  acl::Common::mydie("Simulator is compiling for an unsupport OS!\n");
}

sub get_sim_run_tcl {
  return $fname_runscript;
}

sub get_sim_package {
  return @sim_package;
}

sub get_sim_options {
  return "sim_options.txt";
}

sub qsys_uses_qrun_flow() {
  my $quartus_version = acl::AOCDriverCommon::get_quartus_version_str();
  if (
    $quartus_version =~ m{
      (              # Options group
        1[0-9]|      # 10-19
        2[0-3]       # 20-23
      )              # End options group
      [.]            # Major/Minor separator
    }smx
  ) {
    return 0;
  } else {
    return 1;
  }
}

{
  my $vsim_version_string = undef;

  sub query_raw_vsim_version_string() {
    if ( !defined $vsim_version_string ) {
      $vsim_version_string = `vsim -version`;
      my $error_code = $?;

      if ( $error_code != 0 ) {
        acl::Common::mydie( "Error accessing ModelSim."
            . "  Please ensure you have a valid ModelSim installation on your path.\n"
            . "       Check your ModelSim installation with \"vsim -version\" \n"
        );
      }
    }
    return $vsim_version_string;
  }
}

{
  my $vsim_simple_str = undef;

  sub query_vsim_version_string() {
    if ( !defined $vsim_simple_str ) {
      $vsim_simple_str = query_raw_vsim_version_string();
      $vsim_simple_str =~ s/^\s+|\s+$//g;
    }
    return $vsim_simple_str;
  }
}

{
  my $cosim_64bit = undef;

  sub query_vsim_arch() {
    if ( !defined $cosim_64bit ) {
      my $vsim_version_str = query_raw_vsim_version_string();
      $cosim_64bit = ( $vsim_version_str =~ /64 vsim/ );
    }
    return $cosim_64bit;
  }
}

# For Linux, create a simulation run script with an updated library path that's relative
# to where the host or a.out invoke it
# For Windows, same script is used, but need to remove work_lib as Windows does not support alias to the same work folder
sub _create_run_msim_setup_file($) {
  my $filepath = shift;
  my $file = $filepath.'/mentor/msim_setup.tcl';
  open(FILE, "<$file") or acl::Common::mydie("Can't open $file for read");
  my @lines;
  while(my $line = <FILE>) {
    if ( acl::Env::is_windows() ) {
      # case:595543 remove work_lib from logical_lib for Windows VMAP does not support 2 variables map to same work folder
      # also work_lib is never being used
      $line =~ s|\"work_lib\"||g;
    }
    # 14022161977 Running vsim -version in simulator command line interface (vsim -batch)
    # somehow causes the consecutive commands to just return the simulator version string
    # So use the modelsim inherit command vsimVersionString in place of vsim -version
    $line =~ s|vsim -version|vsimVersionString|g;
    push(@lines,$line);
  }
  close(FILE);
  # Override the original file in case there is any changes
  open(OFH,">$file") or acl::Common::mydie("Can't open $file for write");
  foreach my $line (@lines) {
    print OFH $line;
  }
  close(OFH);
  return 0;
}

sub get_dut_name {
  my ($search_dir, $dut) = @_;
  my $mangled_names;
  my $DUT_HW_TCL_FILE;
  open( DUT_HW_TCL_FILE, "<${search_dir}/${dut}_sys_hw.tcl" )
    or acl::Common::mydie("Couldn't open ${dut}_sys_hw.tcl for read!");
  while ( my $var = <DUT_HW_TCL_FILE> ) {
    if ( $var =~ /ipa\.mangled\.names { (.*) }/ ) {
      $mangled_names = $1;
      last
    }
  }
  close DUT_HW_TCL_FILE;
  return $mangled_names;
}

sub get_dut_avalon_interfaces {
  my ($search_dir, $dut) = @_;
  my @avalon_interfaces;
  my $DUT_SYSTEM_FILE;
  open( DUT_SYSTEM_FILE, "<${search_dir}/${dut}_di.sv" )
    or acl::Common::mydie("Couldn't open ${dut}_di.sv for read!");
  while ( my $var = <DUT_SYSTEM_FILE> ) {
    if ( $var =~ /AV[MS] (.*)/ ) {
      push( @avalon_interfaces, $1 );
    }
    last if ( $var =~ /^\);$/ );
  }
  close DUT_SYSTEM_FILE;
  return (\@avalon_interfaces);
}

=head2 generate_simulation_scripts()

This module creates 3 files shown below. The function is expected to be called in the same folder as <sim>.qsys
 msim_compile.tcl     : The string run by the compilation phase, in the sim script dir
 msim_run.tcl         : The string run by the simulation phase, in the sim script dir
 compile.sh           : The scripts the allows user to recompile their design
Prerequisites:
 $sim_dir             : Directory which the compiled libraries would be created
 $sim_script_dir      : Directory which the simulations scripts reside
 $cosim_lib           : Suffix of cosim folder: [aoc_cosim_msim|hls_cosim_msim]
 $sim_debug           : if defined, Add "log -r *" to msim_run.tcl
 $sim_enable_warnings : if set to 1, it will not suppress compile and elaboration warnings

=cut

sub generate_simulation_scripts($@) {
    # list of kernel/component names for logging
    my @dut_list = @_;
    # vsim version
    my $vsim_version_string = query_vsim_version_string();

    # Library names
    # OpenCL specific setups
    # OpenCL cosim libraries
    my $cosimlib = query_vsim_arch() ? $cosim_lib : $cosim_lib.'32';
    # Script filenames
    my $fname_compilescript = $sim_script_dir.'/'.$sim_compile_script_name;
    # get the whole path for HLS since simulator can be launch outside of the i++.pl without the entry wrapper
    my $sdk_root_path = ($cosim_lib =~ /hls/) ? acl::Env::sdk_root() : "\$::env(".acl::Env::sdk_root_name().")";
    my $fname_svlib = acl::Env::sdk_root() . (acl::Env::is_linux() ? "/host/linux64/lib/lib${cosimlib}" : "/windows64/bin/${cosimlib}");

    # Create the msim_setup_run.tcl script for host to invoke
    _create_run_msim_setup_file(acl::File::abs_path($sim_dir));

    # Generate the modelsim compilation script. Compile is done in $sim_dir
    my $COMPILE_SCRIPT_FILE;
    open(COMPILE_SCRIPT_FILE, ">", $fname_compilescript) or acl::Common::mydie("Couldn't open $fname_compilescript for write!\n");
    print COMPILE_SCRIPT_FILE "onerror {abort all; exit -code 1;}\n";
    print COMPILE_SCRIPT_FILE "set VSIM_VERSION_STR \"$vsim_version_string\"\n";
    print COMPILE_SCRIPT_FILE "set QSYS_SIMDIR $sim_dir\n";
    print COMPILE_SCRIPT_FILE "source \$QSYS_SIMDIR/mentor/msim_setup.tcl\n";
    if ($ENV{'HLS_REGTEST_MODE'}) {
      # Workaround for CASE:14013204663
      print COMPILE_SCRIPT_FILE "set USER_DEFINED_VERILOG_COMPILE_OPTIONS \"+incdir+${sdk_root_path}/ip +define+COSIM_LIB";
    } else {
      print COMPILE_SCRIPT_FILE "set USER_DEFINED_VERILOG_COMPILE_OPTIONS \"+incdir+\$QSYS_SIMDIR +define+COSIM_LIB";
    }
    print COMPILE_SCRIPT_FILE " -suppress 2388" if (!$sim_enable_warnings);  # case:569080: duplicate definition
    print COMPILE_SCRIPT_FILE " -suppress 14408"; # Don't error out if we exceed 5000 node limit with starter edition.
    print COMPILE_SCRIPT_FILE " -suppress 7061";  # Allow variables to be driven in an always_ff block, without being driven by any other process.
    print COMPILE_SCRIPT_FILE "\"\n";
    print COMPILE_SCRIPT_FILE "dev_com\n";  # case:532280: remove after incremental simulation compile works
    print COMPILE_SCRIPT_FILE "com\n";
    # OpenCL does not elaborate now, as it may hard code absolute pathnames, and this won't
    # work well on the farm, as they will be missing when we unpack the results.
    # HLS can elaborate for but will run into issue if folder is moved.
    if ($sim_enable_warnings) {
      print COMPILE_SCRIPT_FILE "if {\$tcl_platform(platform) == \"windows\"} {\n";
      print COMPILE_SCRIPT_FILE "  set fname_svlib \"\\\"${sdk_root_path}/windows64/bin/${cosimlib}\\\"\"\n";
      print COMPILE_SCRIPT_FILE "  set fname_svlib [string map { \"\\\\\" \"/\"} \$fname_svlib]\n";
      print COMPILE_SCRIPT_FILE "} else {\n";
      print COMPILE_SCRIPT_FILE "  set fname_svlib \"${sdk_root_path}/host/linux64/lib/lib${cosimlib}\"\n";
      print COMPILE_SCRIPT_FILE "}\n";
      print COMPILE_SCRIPT_FILE "set USER_DEFINED_ELAB_OPTIONS \"";
      print COMPILE_SCRIPT_FILE "-dpioutoftheblue 1 -sv_lib \$fname_svlib";
      if (acl::Env::is_windows()) {
          print COMPILE_SCRIPT_FILE " -nodpiexports";
      }
      print COMPILE_SCRIPT_FILE ($sim_debug ? " -voptargs=+acc\"\n"
                                            : "\"\n");
      print COMPILE_SCRIPT_FILE "elab\n";
    }
    print COMPILE_SCRIPT_FILE "exit -code 0\n";
    close(COMPILE_SCRIPT_FILE);

    # For OpenCL/SYCL/IPA we use the path to the device_image_ip in a few places
    # There is only one "dut" for the flows we deal with here (OpenCL/SYCL/IPA)
    my $single_dut = lc(@dut_list[0]);
    my $dev_ip_hierarchy_path = "${sim_qsys_name}/${kernel_system_inst}/";
    $dev_ip_hierarchy_path .= ($board_qsys ? 'ks' : "${single_dut}_sys");
    $dev_ip_hierarchy_path .= "/${single_dut}_sys/device_image_ip";

    # Generate the run script. Elaboration is done at the parent level, i.e. $work_dir
    my $RUN_SCRIPT_FILE;
    open(RUN_SCRIPT_FILE, ">", $fname_runscript) or acl::Common::mydie("Couldn't open $fname_runscript for write!\n");
    print RUN_SCRIPT_FILE "onerror {abort all; puts stderr \"The simulation process encountered an error and has aborted.\"; exit -code 1;}\n";
    print RUN_SCRIPT_FILE "set VSIM_VERSION_STR \"$vsim_version_string\"\n";
    print RUN_SCRIPT_FILE "set QSYS_SIMDIR ${sim_dir}\n";
    print RUN_SCRIPT_FILE "if {\$tcl_platform(platform) == \"windows\"} {\n";
    print RUN_SCRIPT_FILE "  set fname_svlib \"\\\"${sdk_root_path}/windows64/bin/${cosimlib}\\\"\"\n";
    print RUN_SCRIPT_FILE "  set fname_svlib [string map { \"\\\\\" \"/\"} \$fname_svlib]\n";
    print RUN_SCRIPT_FILE "} else {\n";
    print RUN_SCRIPT_FILE "  set fname_svlib \"${sdk_root_path}/host/linux64/lib/lib${cosimlib}\"\n";
    print RUN_SCRIPT_FILE "}\n";
    print RUN_SCRIPT_FILE "source \$QSYS_SIMDIR/mentor/msim_setup.tcl\n";
    print RUN_SCRIPT_FILE "# Suppress warnings from the std arithmetic libraries\n";
    print RUN_SCRIPT_FILE "set StdArithNoWarnings 1\n";
    # TODO: case:535241 OpenCL simulator vector_add example design Unknown formal identifier
    # case:1409734405 multiply driven signals generated by system integrator
    #       remove error suppression when all RTL generated is clean
    print RUN_SCRIPT_FILE "set USER_DEFINED_ELAB_OPTIONS \"+nowarnTFMPC ";
    print RUN_SCRIPT_FILE "+nowarnBSOB -suppress 1130 -suppress 2732 -suppress 3584 -suppress 3839 -suppress 12027 -suppress 14408 " if ( ! $sim_enable_warnings );
    if ($sim_user_elab_options) {
      print RUN_SCRIPT_FILE $sim_user_elab_options." ";
    }
    print RUN_SCRIPT_FILE "-dpioutoftheblue 1 -sv_lib \$fname_svlib";
    if (acl::Env::is_windows()) {
        print RUN_SCRIPT_FILE " -nodpiexports";
        print RUN_SCRIPT_FILE " -wlf vsim.wlf" if ( $sim_debug && $cosim_lib =~ /hls/ );
    }
    print RUN_SCRIPT_FILE ($sim_debug ? " -voptargs=+acc\"\n"
                                      : "\"\n");
    print RUN_SCRIPT_FILE "elab\n";
    print RUN_SCRIPT_FILE "onfinish {stop}\n";
    print RUN_SCRIPT_FILE "quietly set StdArithNoWarnings 1\n";
    if ($sim_debug) {
      # (no ghdl) : highest simulation optimization
      # ghdl      : means log all
      # ghdl[=N]  : means log up to a certain level
      print RUN_SCRIPT_FILE "set WLFFilename \$env(WLF_NAME)\n" if ( $cosim_lib =~ /aoc/ );
      if (defined($sim_debug_depth)) {
        if($cosim_lib =~ /hls/ ){
          my $capture_depth = $sim_debug_depth-1;
          my %lc_dut_name_seen;
          foreach my $dut (@dut_list) {
              # Check for DUTs with identical names when canonicalized to lowercase.
              if ($lc_dut_name_seen{lc(${dut})}++) {
                print "WARNING: Component with name '$dut' may not be visible in waveforms because\n".
                  "there is another component with a similar name.  Consider logging all waveforms\n".
                  "to avoid this issue.\n";
              }
              print RUN_SCRIPT_FILE "log -r ".lc(${dut})."_inst/* -depth $capture_depth \n";
          }
        } else {
          # OpenCL/SYCL/IPA
          my $capture_depth = $sim_debug_depth-1;
          print RUN_SCRIPT_FILE "log -r -depth $capture_depth ${dev_ip_hierarchy_path}/*\n";
        }
      }
      else {
        print RUN_SCRIPT_FILE "log -r *\n";
      }
    }
    print RUN_SCRIPT_FILE "run -all\n";
    print RUN_SCRIPT_FILE "set failed [expr [coverage attribute -name TESTSTATUS -concise] > 1]\n";
    print RUN_SCRIPT_FILE "exit -code \${failed}\n";
    close(RUN_SCRIPT_FILE);

    # Generate a script that we'll call to compile the design
    my $EXE_COM_FILE;
    open(EXE_COM_FILE, '>', "$fname_exe_com_script") or acl::Common::mydie("Could not open file '$fname_exe_com_script' $!");
    if (acl::Env::is_linux()) {
      print EXE_COM_FILE "#!/bin/sh\n";
      print EXE_COM_FILE "\n";
      print EXE_COM_FILE "vsim -batch -do \"$fname_compilescript\"\n";
      print EXE_COM_FILE "retval=\$?\n";
      print EXE_COM_FILE "exit \${retval}\n";
    } elsif (acl::Env::is_windows()) {
      print EXE_COM_FILE "vsim -batch -do \"$fname_compilescript\"\n";
      print EXE_COM_FILE "set exitCode=%ERRORLEVEL%\n";
      print EXE_COM_FILE "exit /b %exitCode%\n";
    } else {
      acl::Common::mydie("Unsupported OS detected\n");
    }
    close(EXE_COM_FILE);
    if(acl::Env::is_linux()) {
      system("chmod +x $fname_exe_com_script");
    }

    # Create scripts to use for viewing the waveforms for flows other than HLS
    if ($sim_debug && !($cosim_lib =~ /hls/)) {
      # There is only one "dut" for the flows we deal with here (OpenCL/SYCL/IPA)
      my $dut = @dut_list[0];
      my $prj_dir = acl::File::mydirname($fname_view_do_script);

      # dut_name_map is a string containing all the kernel RTL names separated by
      # spaces, it is expanded during simulation do file generation to instantiate
      # the kernel_name_map array.
      my $dut_name_map = get_dut_name($prj_dir, $dut);

      my $dut_avalon_interface_ref = get_dut_avalon_interfaces($prj_dir, $dut);
      my $avalon_interfaces = join(' ', @$dut_avalon_interface_ref);

      open(VIEW_DO_FILE, '>', "$fname_view_do_script") or acl::Common::mydie("Could not open file '$fname_view_do_script' $!");
      print VIEW_DO_FILE <<END_DO_FILE;
# Only show leaf names in Wave window
config wave -signalnamewidth 1

# Configure wave view column widths
config wave -namecolwidth  350
config wave -valuecolwidth 150

# Navigate to the device_image_ip for $dut
env ${dev_ip_hierarchy_path}

# Add all device_image_ip 'port' signals to the Wave window
set avalon_interfaces { ${avalon_interfaces} }
set all_di_ports [ find signals -ports * ]
set kernel_name_map { ${dut_name_map} }

# Add non-Avalon interface ports
foreach di_port \$all_di_ports {
  set di_leaf_name [file tail \$di_port]
  set add_port 1
  foreach avalon_iface \$avalon_interfaces {
    if { [string first \$avalon_iface \$di_leaf_name] != -1 } {
      set add_port 0
      break
    }
  }
  if { \$add_port == 1 } {
    set added 0
    # Group streaming arguments by component name
    foreach mangled_name \$kernel_name_map {
      if { [string first \$mangled_name \$di_leaf_name] != -1 } {
        if { [string first \${mangled_name}_arg_ \$di_leaf_name] == 0 } {
          set added 1
          add wave -noupdate -expand -group "\$mangled_name" -expand -group Arguments \$di_port
        } elseif { [string first \${mangled_name}_streaming_ \$di_leaf_name] == 0 } {
          set added 1
          add wave -noupdate -expand -group "\$mangled_name" -expand -group Control \$di_port
        } else {
          # This mangled name belongs to a kernel with name substring of this port.
          # It should be added in one of the later iterations, if it is not added
          # then it will be added without grouping.
        }
      }
    }
    # Add all other signals individually
    if { \$added == 0 } {
      add wave -noupdate \$di_port
    }
  }
}

# Add Avalon interfaces, each in its own group
foreach avalon_iface \$avalon_interfaces {
  add wave -noupdate -expand -group \${avalon_iface} \${avalon_iface}*
}

# All waveforms have been added - update the wave window
update

# Zoom to full timeframe
wave zoom full

# Focus on the waveform window
view wave
END_DO_FILE
      close(VIEW_DO_FILE);

      open(VIEW_SYS_FILE, '>', "$fname_view_sys_script") or acl::Common::mydie("Could not open file '$fname_view_sys_script' $!");
      # OpenCL simulation puts the vsim.wlf file one directory higher
      my $extra_path_for_opencl_l = ($sycl_mode ? '' : '../');
      my $extra_path_for_opencl_w = ($sycl_mode ? '' : '..\\');
      if (acl::Env::is_linux()) {
        print VIEW_SYS_FILE <<END_LINUX_SH_FILE;
#!/bin/sh
# Identify the directory from which to run
rundir=\$PWD
scripthome=\$( cd -- "\$( dirname -- "\${BASH_SOURCE[0]}" )" &> /dev/null && pwd )
# Ensure the waveform file exists
if [ ! -e \${scripthome}/${extra_path_for_opencl_l}vsim.wlf ] ; then
  echo
  echo "ERROR: Waveform file not found.  Have you run the design?"
  echo
  exit 1
fi
# Create the sim directory in case it was cleaned up, and cd into it
mkdir -p \${scripthome}/mpsim/sim
cd \${scripthome}/mpsim/sim
# Start the vsim GUI
vsim -view ../../${extra_path_for_opencl_l}vsim.wlf -do ../../vsim.do
retval=\$?
cd \${rundir}
return \${retval} &> /dev/null || exit \${retval}
END_LINUX_SH_FILE
      } elsif (acl::Env::is_windows()) {
        print VIEW_SYS_FILE <<END_WINDOWS_CMD_FILE;
\@echo off
REM Identify the directory from which to run
set rundir=%cd%
set scripthome=%~dp0
REM Ensure the waveform file exists
if NOT exist "%scripthome%\\${extra_path_for_opencl_w}vsim.wlf" (
  echo:
  echo ERROR: Waveform file not found.  Have you run the design?
  echo:
  exit /b 1
)
REM Create the sim directory in case it was cleaned up, and cd into it
md "%scripthome%\\mpsim\\sim" 2> nul
cd "%scripthome%\\mpsim\\sim"
REM Start the vsim GUI
vsim -view ..\\..\\${extra_path_for_opencl_w}vsim.wlf -do ..\\..\\vsim.do
set exitCode=%ERRORLEVEL%
cd %rundir%
exit /b %exitCode%
END_WINDOWS_CMD_FILE
      } else {
        acl::Common::mydie("Unsupported OS detected\n");
      }
      close(VIEW_SYS_FILE);

      # FIXME: This doesn't work because the icpx driver resets the permissions
      # when it copies the file from the temp dir to the final project directory
      #chmod 0740, $fname_view_sys_script if acl::Env::is_linux();
    }
    return 0;
}

# Write out a the memory copy function file at the same location as compile to specified how host memory transfer should be done, default is bypass memory controller.
# User can update this file after a compile is complete to overwrite the transfer option to go through  memory controller
# As a workaround, it generate a dummy MACRO for EMIF instance path instead of the real one
# example of a real s10gx
# `define EMIF_BANK0_MEM tb.board_inst.acl_ddr4_s10.acl_ddr4_s10_emif_s10_0.acl_ddr4_s10_emif_s10_0.arch.arch_inst.io_tiles_wrap_inst.abphy_tiles.io_tiles_abphy_inst.mem_array_abphy_inst.mem
sub _generate_memory_copy_function($) {
  my $out_dir = shift;
  my $mem_transfer_options_filename = "aoc_sim_mcf_options.svh";
  # The OpenCL memory copy function options is manually written in 18.1 and before because EMIF IP does not have this feature available
  # User can overwrite this when they create their own function and transfer options
  return if (-e "$out_dir/$mem_transfer_options_filename");
  # Data transfer through memory controller is only meaningful to use with accurate memory
  my $sim_mem_ctrl = ($sim_accurate_memory == 1) ? 1 : 0;
  open(FH, ">$out_dir/$mem_transfer_options_filename") or acl::Common::mydie("Cannot open $sim_dir/$mem_transfer_options_filename for write\n");
  # Default is load and read data bypassing EMIF memory controller
  # When set to 1, it means host-device transfer is through use memory bank divider to EMIF. This is the default support for 19.1
  # The reason to have an option was designed to help isolate host-device transfer issue on Silicon. See Case:533638: "example_designs/multithread_vector_operation HW run failure on S10 due
  # to result mismatch cause by apparent PCIe DMA bug" for detail description. The way the failure was isolate was use the simulator to prove bank divider and EMIF was working properly.
  # Therefore the only issue remained the PCIe DMA IP.
  # When set to 0, it means host-device transfer is through bypassing the memory bank divider. Once EMIF adds the feature, then this should be updated to always set to 0.
  # Case:525522: "Figure out EMIF unsupported temporary solution to backdoor load memory" is the request to add the support assigned to EMIF IP team.
  print FH "  localparam MODEL_HOST_MEMORY_READ      = ${sim_mem_ctrl};\n";
  print FH "  localparam MODEL_HOST_MEMORY_WRITE     = ${sim_mem_ctrl};\n";
  # In 18.1 and before, EMIF does not support HMC bypass mode, and so we used a macro and function to get translate AVL address to memory storage access
  print FH "`define EMIF_BANK0_MEM dummy_mem\n";
  print FH "  function bit [ABPHY_ADDR_W-1:0] get_emif_abphy_addr(input bit [63:0] addr);\n";
  print FH "    get_emif_abphy_addr = {{ABPHY_ADDR_W}{1'b0}};\n";
  print FH "  endfunction\n";
  close(FH);
}

# generate the include file matching in aoc_fake_pll.sv to overwrite predefined frequency
sub generate_set_sim_freq_file {
  # convert frequency to 2X clock period
  # 2x clock half period = 1/sim_kernel_clk_frequency(MHz) X 1E6 ps/us / 4
  my $clock2x_half_period = int(250000/$sim_kernel_clk_frequency);
  my $fake_pll_inc_filename = "$sim_dir/set_sim_freq.svh";

  # variable name "clock2x_half_period_ps" must be consistent with opencl_sim/aoc_fake_pll IP
  # make sure the file name is same as aoc_fake_pll.sv

  open(FH, ">$fake_pll_inc_filename") or acl::Common::mydie("Cannot open $fake_pll_inc_filename\n");
  print FH "localparam real clock2x_half_period_ps = $clock2x_half_period;\n";
  close(FH);
  return;
}

# Write a verilog and vhdl filelist
# vhdl is also supported for AOC back in 18.0 when DSPBA generates VHDL
sub _generate_kernel_system_filelist() {
  my @sv_list = ();
  my @vhdl_list = ();
  my @acl_lib_list = ();

  if ($kernel_system_qsys) {
    # kernel_system.tcl was invoked, qsys-generate files in ip/kernel_system and kernel_system
    # Parse the kernel_system/*.qip + ip/kernel_system/*.qip to get the kernel_system files
    append_rtl_filelist_recur("ip/kernel_system", \@sv_list, \@vhdl_list, \@acl_lib_list);
    append_rtl_filelist_recur("kernel_system", \@sv_list, \@vhdl_list, \@acl_lib_list);
  }
  else {
    # Only OpenCL has QSys-less flow and places files in kernel_hdl, parse the kernel_system.qip
    push @sim_package, 'kernel_hdl';
    # parse the kernel_system.qip
    _append_rtl_filelist_from_qip("kernel_system.qip", '.', \@sv_list, \@vhdl_list, \@acl_lib_list);
  }
  my $qsys_sim_tcl_script = "modelsim_files.tcl";
  # Not sure why when using Script to create it, it prepends qsys_name, but it doesn't do that when I use GUI
  my $sim_files_tcl_fname = "ip/${sim_qsys_name}/${sim_qsys_name}_${kernel_system_inst}/sim/common/".$qsys_sim_tcl_script;
  if (! -e $sim_files_tcl_fname) {
    $sim_files_tcl_fname = "ip/${sim_qsys_name}/${kernel_system_inst}/sim/common/".$qsys_sim_tcl_script;
    if (! -e $sim_files_tcl_fname) {
      acl::Common::mydie("Cannot find $sim_files_tcl_fname.\n");
    }
  }
  # remove previous design files
  my @sim_script_text = ();
  my $in_design_lib_proc = 0;
  my $in_design_files_proc = 0;
  open(FH, "$sim_files_tcl_fname") or acl::Common::mydie("Can't open $sim_files_tcl_fname for read.\n");
  while (<FH>) {
    my $text =$_;
    if ($in_design_files_proc || $in_design_lib_proc) {
      if ($text =~ /^\s+(lappend.*kernel_system\.sv|dict set)/) {
        next;  # do not append previous empty shell kernel_system.sv
      }
      elsif ($text =~ /return/) {
        # reset variable
        $in_design_lib_proc = 0;
        $in_design_files_proc = 0;
      }
    }
    elsif ($text =~ /get_design_libraries/) {
      $in_design_lib_proc = 1;
    }
    elsif ($text =~ /get_design_files/) {
      $in_design_files_proc = 1;
    }
    push(@sim_script_text, $_);
  }
  close(FH);
  $in_design_lib_proc = 0;
  $in_design_files_proc = 0;
  # add acl_ip library and kernel_system files
  my $sdk_root_name = acl::Env::sdk_root_name();
  my $acl_ip_lib = "acl_ip";
  my $kernel_system_lib = $sim_qsys_name."_".$kernel_system_inst;  # name match with what QSys created
  my $qsys_compile_prefix = "    lappend design_files ";
  my $qsys_uses_qrun = qsys_uses_qrun_flow();
  my $qsys_include_dir = "\$::env(${sdk_root_name})/ip";
  open(FH, ">$sim_files_tcl_fname") or acl::Common::mydie("Can't open $sim_files_tcl_fname for write.\n");
  for my $text (@sim_script_text) {
    if ($in_design_lib_proc) {
      print FH $text;  # prints initialization
      print FH "    dict set libraries $acl_ip_lib 1\n";
      print FH "    dict set libraries $kernel_system_lib 1\n";
      $in_design_lib_proc = 0;
      next;
    }
    elsif ($in_design_files_proc) {
      print FH $text;  # prints initialization, i.e. set design files [list]
      if (acl::Env::is_windows()) {
        # For Windows, INTELFPGAOCLSDKROOT contains forward slash which gets translated a escape characters by Modelsim
        print FH '    set intelfpgaoclroot $::env(INTELFPGAOCLSDKROOT)'."\n";
        print FH '    set intelfpgaoclroot [string map {\\\\ /} $intelfpgaoclroot]'."\n";
        $qsys_include_dir = "\$intelfpgaoclroot/ip";
      }
      foreach my $acl_ip (@acl_lib_list) {
        if ($acl_ip =~ /\.vhd$/) {
          if ($qsys_uses_qrun) {
            print FH "$qsys_compile_prefix\"-makelib $acl_ip_lib $acl_ip   -end\"\n";
          } else {
            print FH "$qsys_compile_prefix\"vcom $acl_ip    -work $acl_ip_lib\"\n";
          }
        }
        else {
          my $acl_ip_processed = $acl_ip;
          if (acl::Env::is_windows()) {
            $acl_ip_processed =~ s/\$::env\(INTELFPGAOCLSDKROOT\)/\$intelfpgaoclroot/g;
          }
          if ($qsys_uses_qrun) {
            print FH "$qsys_compile_prefix\"-vlog.options +incdir+$qsys_include_dir -end -makelib acl_ip $acl_ip_processed   -end\"\n";
          } else {
            print FH "$qsys_compile_prefix\"vlog -sv \$USER_DEFINED_VERILOG_COMPILE_OPTIONS \$USER_DEFINED_COMPILE_OPTIONS +incdir+$qsys_include_dir $acl_ip_processed    -work acl_ip\"\n";
          }
        }
      }
      foreach my $sv_filepath (@sv_list) {
        if ($qsys_uses_qrun) {
          print FH "$qsys_compile_prefix\"-makelib $kernel_system_lib \\\"[normalize_path \"\$QSYS_SIMDIR/../../../../$sv_filepath\"]\\\"   -end\"\n";
        } else {
          print FH "$qsys_compile_prefix\"vlog -sv \$USER_DEFINED_VERILOG_COMPILE_OPTIONS \\\"[normalize_path \"\$QSYS_SIMDIR/../../../../$sv_filepath\"]\\\"  -work $kernel_system_lib\"\n";
        }
      }
      foreach my $vhdl_filepath (@vhdl_list) {
        if ($qsys_uses_qrun) {
          print FH "$qsys_compile_prefix\"-makelib $kernel_system_lib \\\"[normalize_path \"\$QSYS_SIMDIR/../../../../$vhdl_filepath\"]\\\"   -end\"\n";
        } else {
          print FH "$qsys_compile_prefix\"vcom \$USER_DEFINED_VHDL_COMPILE_OPTIONS \\\"[normalize_path \"\$QSYS_SIMDIR/../../../../$vhdl_filepath\"]\\\"  -work $kernel_system_lib\"\n";
        }
      }
      $in_design_files_proc = 0;
      next;
    }
    elsif ($text =~ /get_design_libraries/) {
      $in_design_lib_proc = 1;
    }
    elsif ($text =~ /get_design_files/) {
      $in_design_files_proc = 1;
    }
    print FH $text;
  }
  close(FH);
  return;
}

# recursively and find all the qip files
sub append_rtl_filelist_recur($$$$) {
  my ($path, $sv_list, $vhdl_list, $acl_lib_list) = @_;

  # resursively find all the qip files in the folder
  opendir(DIR, $path) or acl::Common::mydie("Cannot open $path\n");
  my @files = readdir(DIR);
  closedir(DIR);
  foreach my $file (@files) {
    next if ($file =~ /^\.\.?$/);
    my $filepath = $path."/".$file;

    if ($file =~ /\.qip$/) {
      my $qip_file = $path."/".$file;
      _append_rtl_filelist_from_qip($qip_file, $path, $sv_list, $vhdl_list, $acl_lib_list);
    }
    elsif (-d $filepath) {
      append_rtl_filelist_recur($filepath, $sv_list, $vhdl_list, $acl_lib_list);
    }
  }
}

sub _append_rtl_filelist_from_qsf($$$$) {
  my ($qsf_file, $sv_list, $vhdl_list, $acl_lib_list) = @_;
  my $verbose = acl::Common::get_verbose();

  open(FILE, "<$qsf_file") or acl::Common::mydie("Can't open $qsf_file for read\n");
  while(<FILE>) {
    my $line = $_;
    if ($line =~ /(VERILOG_FILE|MISC_FILE)\s+(\S+)/) {
      my $language = $1;
      my $filepath = $2;
      if ($language eq "MISC_FILE") {
        # ignore Header files, .ip files and etc
        if ($filepath =~ /hex/) {
          # copy hex files over to $
          my $hex_file = acl::File::mybasename($filepath);
          acl::File::copy($filepath, $hex_file);
          push @sim_package, $hex_file;
        }
        next;
      }
      elsif ($filepath =~ /\$::env\S+\).*/) {
        # Library file. Modelsim is already tcl based, so path is identical
        push @{$acl_lib_list}, $filepath;
        next;
      }
    }
    elsif ($line =~ /QIP_FILE\s+(\S+)/) {
      my $qip_file = $1;
      _append_rtl_filelist_from_qip($qip_file, ".", $sv_list, $vhdl_list, $acl_lib_list);
    }
  }
  close(FILE);
}

sub _append_rtl_filelist_from_qip($$$$$) {
  my ($qip_file, $path, $sv_list, $vhdl_list, $acl_lib_list) = @_;
  my $verbose = acl::Common::get_verbose();

  open(FILE, "<$qip_file") or acl::Common::mydie("Can't open $qip_file for read\n");
  while(<FILE>) {
    my $line = $_;
    if ($line =~ /(VERILOG_FILE|VHDL_FILE|MISC_FILE).*qip_path.*\"(.*)\"/) {
      my $language = $1;
      my $filepath = $2;
      if ($language eq "MISC_FILE") {
        # ignore Header files, .ip files and etc
        if ($filepath =~ /hex/) {
          # copy hex files over to $
          my $hex_file = acl::File::mybasename($filepath);
          acl::File::copy($path."/".$filepath, $hex_file);
          push @sim_package, $hex_file;
        }
        next;
      }
      elsif ($filepath =~ /\$::env\S+\).*/) {
        # Library file. Modelsim is already tcl based, so path is identical
        push @{$acl_lib_list}, $filepath;
        next;
      }
      # a Verilog/Systemverilog or VHDL file set pointer to the right list to populate
      my $list_ptr;
      $list_ptr = ($language eq "VERILOG_FILE") ? $sv_list : $vhdl_list;
      # add the absolute path
      push @{$list_ptr}, $path."/".$filepath;
      # add files that are in workdir
      push @sim_package, $filepath if ($filepath !~ /\// && $path eq ".");
    }
  }
  close(FILE);
}

=head2 compile_opencl_simulator($prog, $fulllog, $work_dir)

Calls com function in msim_setup after simulation scripts is generated. Returns
0 on success.  Prerequisite to calling this function is setup of the simulation
directory variables, i.e. sim_dir, sim_script_dir, kernel_system_inst, etc.

=cut

sub compile_opencl_simulator {
  my ($prog,$fulllog, $work_dir) = @_;
  my $verbose = acl::Common::get_verbose();
  my $quiet_mode = acl::Common::get_quiet_mode();

  # Add ip subfolder in @sim_package for Quartus Pro only
  push @sim_package, "ip" if (!$acl::AOCDriverCommon::using_quartus_std);

  # include the correct directory for package
  push @sim_package, $sim_qsys_name;
  push @sim_package, "kernel_system" if (! $board_qsys);

  # copy the relevant files over
  _opencl_sim_copy_files($work_dir);

  $fname_view_sys_script = acl::Env::is_linux() ? 'view_waveforms.sh' : 'view_waveforms.cmd';
  $fname_view_sys_script = $work_dir.'/'.$fname_view_sys_script;
  $fname_view_do_script = $work_dir.'/vsim.do';
  my $prj_name = acl::File::mybasename($fulllog);
  $prj_name =~ s/\.log$//;

  # set the cosim library for OpenCL
  set_sim_library_to_aoc();
  # That could be from System Integrator in old flow or from calling generate_msim_system_tcl()
  _generate_kernel_system_filelist() if ( $board_qsys );
  # Generate the fake_pll frequency as accurate memory does not run with 1 GHz
  generate_set_sim_freq_file() if ( $sim_accurate_memory );
  # Queries for Modelsim licence and creates compile and run the script
  generate_simulation_scripts(($prj_name));
  print "Compiling simulation...\n" if $verbose && !$quiet_mode;
  my $cmd = get_sim_compile_exe();
  my $return_status = acl::Common::mysystem_full(
   {'time' => 1, 'time-label' => 'compiling-simulation, ', 'stdout' => 'msim-compile.tmp', 'stderr' => '&STDOUT'},
     $cmd);
  if ( $return_status != 0 ) {
    copy_unique_errors_to_log("msim-compile.tmp", '&STDERR');
    acl::Common::move_to_log("!========== msim-compile ==========","msim-compile.tmp",$fulllog);
    acl::Common::mydie("Simulation compile FAILED.\nRefer to ".acl::AOCDriverCommon::get_log_file($fulllog, 1)." for details.\n");
  }
  acl::Common::move_to_log("!========== msim-compile ==========","msim-compile.tmp",$fulllog);
  print "$prog: Simulation generation done!\n" unless $quiet_mode;
  push @sim_package, "modelsim.ini";
  push @sim_package, "libraries";

  # sim dir need to be placed in aocx for aoc_msim_device to find the right location
  my $sim_path_filename = get_sim_options();
  open SIMDIR_FH, ">$sim_path_filename" or acl::Common::mydie("Couldn't open $sim_path_filename for write.\n");
  print SIMDIR_FH $sim_script_dir."\n";
  print SIMDIR_FH $sim_accurate_memory."\n";
  close SIMDIR_FH;

  return 0;
}

sub copy_unique_errors_to_log { #filename, logfile
  my $filename = shift;
  my $logfile= shift;
  open(LOG, ">>$logfile") or mydie("Couldn't open $logfile for appending.");
  open(TMP, "<$filename") or mydie("Couldn't open $filename for reading.");
  my %seen_errs = ();
  while(my $l = <TMP>) {
    if ( $l =~ /Error: (.*)/ ) {
      my $error_string = $1;
      # only keep errors with an error code - the others don't say much useful
      if ( $error_string =~ /\(.*\)/ ) {
        if (not defined $seen_errs{$error_string}) {
          $seen_errs{$error_string} = "1";
        }
      }
    } elsif ( $l =~ /Fatal: (.*)/ ) {
      my $error_string = $1;
      if (not defined $seen_errs{$error_string}) {
        $seen_errs{$error_string} = "1";
      }
    }
  }
  foreach my $key (keys %seen_errs) {
    print LOG "Simulation Error: $key \n";
  }
  close TMP;
  close LOG;
}

sub _opencl_sim_copy_files {
  my $work_dir = shift;
  # Do nothing if QSys generates everything - backward compatible
  return if !$board_qsys;
  # Only copy over hex file if vendor provides accurate memory model or less-QSys flow
  foreach my $d (@sim_package) {
    acl::File::copy_tree($qsys_temp_dir_path."/".$d, $work_dir);
  }
  # Copy hex files (for ROM's) by system integrator into simulation directory
  acl::File::copy("*.hex", $sim_script_dir);
}

=head2 opencl_create_sim_system($bsp_variant, $board_variant, $less_qsys, $work_dir_no_base, $bcxml_file, $fulllog, $work_dir, $in_sycl_mode)

Create the OpenCL simulation system by calling QSys and precompile common library.
This is to create the simulation harness with an empty kernel for QSys-less flow

=cut

sub opencl_create_sim_system($$$$$$$$) {
  my ($bsp_variant, $board_variant, $less_qsys, $work_dir_no_base, $bcxml_file, $fulllog, $work_dir, $in_sycl_mode) = @_;
  my $verbose = acl::Common::get_verbose();
  # set global variables
  $board_qsys = $less_qsys;  # skip_qsys == less_qsys == board_qsys
  $sycl_mode = $in_sycl_mode;
  my $fulllog_abs = $work_dir."/".$fulllog;

  my $board_spec_xml = "$work_dir/board_spec.xml";
  if ( not -e $board_spec_xml ) {
    $board_spec_xml = acl::AOCDriverCommon::find_board_spec_ver_ipauth($bsp_variant,$board_variant);
  }
  my $devicemodel = uc acl::Env::aocl_boardspec( "$board_spec_xml", "devicemodel");
  # Remove trailing '_.*' from devicemodel: a10_sdk -> a10
  ($devicemodel) = $devicemodel =~ /(.*)_.*/;
  my $devicefamily = acl::AOCDriverCommon::device_get_family_no_normalization($devicemodel);
  my $has_snoop =  acl::Env::aocl_boardspec( "$board_spec_xml", "has_snoop");
  my $has_clk2x = acl::Env::aocl_boardspec( "$board_spec_xml", "has_clk2x");

  if ($board_qsys == 0) {
    # flow for 18.0 and 18.1, kept for IO channel
    set_sim_dir_path(".");  # set qsys directory path same as ".", qsys directory name as work directory
    initialize_simulation_paths($opencl_qsys_name, $acl::AOCDriverCommon::using_quartus_std);
    _opencl_sim_generate_system($devicefamily, $devicemodel, $has_snoop, $has_clk2x, $bcxml_file, $fulllog_abs);
    return;
  }
  # check for hidden flag and options for -sim-input-dir
  # In 18.1.1, this feature only supports pre-generated simulator with the QSys name mpsim via aoc_sim_generate_qsys.tcl
  if (defined($qsys_temp_dir_path)) {
    initialize_simulation_paths($opencl_qsys_name, $acl::AOCDriverCommon::using_quartus_std);
    print "Found previous compiled testbench in $qsys_temp_dir_path. Skipping testbench generation.\n" if $verbose;
    return;
  }
  # initialize simulation qsys directory with default directory name when user did not supply one
  $qsys_temp_dir_path = $work_dir_no_base.$qsys_temp_dir;
  if (!$sim_accurate_memory) {
    initialize_simulation_paths($opencl_qsys_name, $acl::AOCDriverCommon::using_quartus_std);
    acl::File::remove_tree($qsys_temp_dir_path) if (-e $qsys_temp_dir_path);
    acl::File::make_path($qsys_temp_dir_path) or acl::Common::mydie("Cannot create temporary simulation directory $qsys_temp_dir: $!");
    acl::Common::mydie("Cannot find $bcxml_file: $!") if (!(-e $bcxml_file));
    chdir $qsys_temp_dir_path or acl::Common::mydie("Cannot change dir into $qsys_temp_dir_path: $!");
    _opencl_sim_generate_system($devicefamily, $devicemodel, $has_snoop, $has_clk2x, $bcxml_file, $fulllog_abs);
    my $orig_dir = acl::Common::get_original_dir();
    chdir $orig_dir or acl::Common::mydie("Cannot change back into directory $orig_dir: $!");
  }
  # TODO: else Future support to be able to build performance accurate memory model during compile time
}

sub get_avalon_interface_mappings {
  my ($search_dir) = @_;
  my $KERNEL_SYSTEM_FILE;
  open( KERNEL_SYSTEM_FILE, "<${search_dir}/kernel_system.sv" )
    or acl::Common::mydie("Couldn't open kernel_system.sv for read!");
  my @host_interfaces;
  my @agent_interfaces;
  while ( my $var = <KERNEL_SYSTEM_FILE> ) {
    if ( $var =~ /AVM (.*)/ ) {
      push( @host_interfaces, $1 );
    }
    if ( $var =~ /AVS (avs_.*)/ ) {
      push( @agent_interfaces, $1 );
    }
    last if ( $var =~ /^\);$/ );
  }
  close KERNEL_SYSTEM_FILE;
  return (\@host_interfaces, \@agent_interfaces);
}

sub get_generated_components {
  # read the comma-separated list of components from a file
  my ($project_bc_xml_filename) = @_;
  my $BC_XML_FILE;
  open( BC_XML_FILE, "<${project_bc_xml_filename}" )
    or acl::Common::mydie("Couldn't open ${project_bc_xml_filename} for read!");
  my @dut_array;
  my @num_cra_args;
  my @stream_intf_mapping;
  my @host_pipe_mapping;
  while ( my $var = <BC_XML_FILE> ) {
    if ( $var =~ /<KERNEL name=".*" kname="(.*)" filename/ ) {
      push( @dut_array, $1 );
      push( @num_cra_args, 0 );
    }
    if ($var =~ /<KERNEL name=".*" kname=".*" filename.*is_streaming="false"/) {
      $num_cra_args[-1] = $num_cra_args[-1] + 1; # CRA control
    }
    if ( $var =~ /<ARGUMENT .* streaming_cra_port="([^\s]*)"/ ) {
      if ( $1 eq '' ) {
        $num_cra_args[-1] = $num_cra_args[-1] + 1;
      } else {
        push( @stream_intf_mapping, $1 );
        push( @stream_intf_mapping, $dut_array[-1] );
      }
    }
    if ( $var =~ /<INTERFACE .* chan_id="(.*)" is_fifo.* host_io_pipe ="true"/ ) {
      push( @host_pipe_mapping, $1 );
      push( @host_pipe_mapping, $dut_array[-1] );
    }

  }
  close BC_XML_FILE;
  return (\@dut_array, \@num_cra_args, \@stream_intf_mapping, \@host_pipe_mapping);
}

# Create a simulation system at compile time, copy over the top level testbench
sub _opencl_sim_generate_system {
  my ($devicefamily, $devicemodel, $use_snoop, $use_clk2x, $bcxml_file, $fulllog) = @_;
  my $verbose = acl::Common::get_verbose();
  my $quiet_mode = acl::Common::get_quiet_mode();
  my $sopc_builder_cmd = "qsys-script";
  my $ip_gen_cmd = 'qsys-generate';

  # Run Java Runtime Engine with max heap size 512MB, and serial garbage collection.
  my $jre_tweaks = "-Xmx512M -XX:+UseSerialGC";

  # Create the complete simulation system for less accurate memory model or not previous created folder
  print "Creating simulation system...\n" if $verbose && !$quiet_mode;
  my $qsys_quartus_project;
  if (!$acl::AOCDriverCommon::using_quartus_std) {
    $qsys_quartus_project = "--quartus-project=none";  # only supports >=18.0 pro version
  } else {
    $qsys_quartus_project = ""; # for Quartus Standard there shouldn't be a project specified
  }
  # Copy over all ipx and iipx files from share/lib folder so Qsys can find the aoc cosim IP's.
  my @ipx_filelist = ("iface.ipx", "hw_iface.iipx", "sw_iface.iipx");
  foreach my $ipx_file (@ipx_filelist) {
    acl::File::copy(acl::Env::sdk_root()."/share/lib/opencl_sim/".$ipx_file, $ipx_file);
  }

  # In case we are using Quartus Standard, iface.ipx needs to contain all HW components
  # (which are in hw_iface.iipx for Quartus Pro)
  if ($acl::AOCDriverCommon::using_quartus_std) {
    acl::File::copy(acl::Env::sdk_root()."/share/lib/opencl_sim/hw_iface.iipx", "iface.ipx");
  }

  if ($board_qsys) {
    acl::Env::create_opencl_ipx(".");
    # Less accurate memory model and -skip-qsys flow uses to stitch together the whole system
    # TODO: case:576491 Use System Integrator to generate empty_kernel_hw.tcl
    #       temporary workaround parse .bc.xml and copy over example kernel system hw tcl
    _create_empty_kernel_system();
  }
  # TODO: cache some information about boardspec.xml to the folder so the simulation framework matches later
  my $generate_system_script = 'aoc_create_msim_system.tcl';
  my ($dut_array_ref, $num_cra_args_array_ref, $stream_intf_mapping_ref, $host_pipe_mapping_ref) = get_generated_components($bcxml_file);
  my ($host_iface_mapping_ref, $agent_iface_mapping_ref) = get_avalon_interface_mappings(acl::File::mydirname($bcxml_file));
  my $HOST_IFACE_LIST = join( ',', @$host_iface_mapping_ref );
  my $AGENT_IFACE_LIST = (@$agent_iface_mapping_ref) ? join( ',', @$agent_iface_mapping_ref ) : "{}";
  my $DUT_LIST = join( ',', @$dut_array_ref );
  my $AGENT_INTERFACES = join ( ',', @$num_cra_args_array_ref);
  my $HOST_PIPE_MAPPING = (@$host_pipe_mapping_ref) ? join ( ',', @$host_pipe_mapping_ref) : "{}";
  my $STREAM_INTF_MAPPING = (@$stream_intf_mapping_ref) ? join ( ',', @$stream_intf_mapping_ref) : "{}";
  my $init_var_tcl_cmd = ($DUT_LIST ne '') ? "--cmd=\"set component_list $DUT_LIST; set num_agent_interface $AGENT_INTERFACES; set host_iface_list $HOST_IFACE_LIST; set agent_iface_list $AGENT_IFACE_LIST; set stream_intf_mapping $STREAM_INTF_MAPPING; set host_pipe_mapping $HOST_PIPE_MAPPING; set sycl_mode $sycl_mode\"" : '';
  _generate_msim_system_wrapper_tcl($devicefamily, $use_snoop, $use_clk2x, $bcxml_file, $generate_system_script);
  my $return_status = acl::Common::mysystem_full(
    {'time' => 1, 'time-label' => 'qsys-script-simulation, ', 'stdout' => 'qsys-script.tmp', 'stderr' => '&STDOUT'},
    "$sopc_builder_cmd $qsys_quartus_project --script=$generate_system_script $jre_tweaks $init_var_tcl_cmd");
  acl::Common::move_to_log("!========== create-simulation ==========","qsys-script.tmp",$fulllog);
  $return_status == 0 or acl::Common::mydie("Simulation system creation FAILED.\nRefer to ".acl::AOCDriverCommon::get_log_file($fulllog,1)." for details.\n");

  # If ipx files are not present, generate them
  if (! -e "iface.ipx") {
      print "Generating ipx...\n" if $verbose && !$quiet_mode;
  $return_status = acl::Common::mysystem_full(
    {'time' => 1, 'time-label' => 'qsys-generate-ipx, ', 'stdout' => 'ipxgen.tmp', 'stderr' => '&STDOUT'},
    "ip-make-ipx --source-directory=\"".acl::Env::sdk_root()."/ip/board\" --output=iface.ipx  --relative-vars=INTELFPGAOCLSDKROOT");
  }

  # Generate the simulatable HDL
  print "Generating simulation system...\n" if $verbose && !$quiet_mode;
  my $device_model_part_info = ''; # Provide --part=* for devices that require specifying when there is no qsys default
  if ($devicefamily =~ /^Agilex\s*5$/ismx) { # Early device support for Agilex 5, specify part number
    $device_model_part_info = "--part=\"$devicemodel\"";
  }

  # Obtain quartus version for backwards compatibility
  my $quartus_version = acl::AOCDriverCommon::get_quartus_version_str();
  # Supporting backwards compatibility for Agilex rebranding to Agilex 7 for
  # versions of Quartus before 23.1
  if ($devicefamily =~ m/^Agilex\s*7$/ismx){
    if (
      $quartus_version =~ m{
        (              # Options group
          1[0-9]|      # 10-19
          2[0-2]       # 20-22
        )              # End options group
        [.]            # Major/Minor separator
      }smx
    ) {
      $devicefamily = 'Agilex';
    }
  }

  my $modelsim_flow = '--modelsim-flow=traditional';
  if ( 
    (defined $sim_use_qrun_flow) ||
    ($quartus_version =~ m{
      (              # Options group
        1[0-9]|      # 10-19
        2[0-3]       # 20-23
      )              # End options group
      [.]            # Major/Minor separator
    }smx)
  ) {
    # modelsim-flow option is absent for versions of Quartus before 24.1
    $modelsim_flow = '';
  }

  $return_status = acl::Common::mysystem_full(
    {'time' => 1, 'time-label' => 'qsys-generate-simulation, ', 'stdout' => 'ipgen.tmp', 'stderr' => '&STDOUT'},
    "$ip_gen_cmd $modelsim_flow --family=\"$devicefamily\" $device_model_part_info ${sim_qsys_name}.qsys --simulation $qsys_quartus_project --jvm-max-heap-size=3G --clear-output-directory");
  acl::Common::move_to_log("!========== generate-simulation ==========","ipgen.tmp",$fulllog);
  $return_status == 0 or acl::Common::mydie("Simulation system generation FAILED.\nRefer to ".acl::AOCDriverCommon::get_log_file($fulllog,1)." for details.\n");

  # In case of Quartus Standard we also need to generate kernel_system.qsys
  # In Quartus Pro subsystems get automatically generated, not in Quartus Standard
  if($acl::AOCDriverCommon::using_quartus_std) {
    $return_status = acl::Common::mysystem_full(
      {'time' => 1, 'time-label' => 'qsys-generate-simulation-kernel-system, ', 'stdout' => 'ipgen.tmp', 'stderr' => '&STDOUT'},
      "$ip_gen_cmd $modelsim_flow --family=\"$devicefamily\" kernel_system.qsys --simulation $qsys_quartus_project --jvm-max-heap-size=3G --clear-output-directory");
    acl::Common::move_to_log("!========== generate-simulation kernel_system ==========","ipgen.tmp",$fulllog);
    $return_status == 0 or acl::Common::mydie("Simulation system for kernel_system generation FAILED.\nRefer to ".acl::AOCDriverCommon::get_log_file($fulllog,1)." for details.\n");
  };

  # Write the emif access script to determine how EMIF write and read from host
  _generate_memory_copy_function($sim_script_dir);
}

sub _create_empty_kernel_system() {
  # TODO: call system integrator to generate the empty_kernel_system_hw.tcl file
  my $empty_ks_script = acl::Env::sdk_root()."/share/lib/opencl_sim/empty_kernel_system_hw.tcl";
  my $kernel_system_rtl = acl::Env::sdk_root()."/share/lib/opencl_sim/kernel_system.sv";
  acl::File::copy($empty_ks_script, "empty_kernel_system_hw.tcl");
  acl::File::copy($kernel_system_rtl, "kernel_system.sv") if (! (-e "kernel_system.sv"));  # don't over existing one
}

sub _generate_msim_system_wrapper_tcl {
  my ($devicefamily, $use_snoop, $use_clk2x, $bcxml_file, $outfile) = @_;
  my $qsys_generate_system_script = acl::Env::sdk_root()."/share/lib/tcl/aoc_sim_generate_qsys.tcl";
  # TODO: case:520397 Enable Interleave slave bfm memory
  #       open board spec and get memory size, interleaving, and interleaving size if applicable
  #       set the variables then source the script
  my $host_to_dev_width = 0;
  my $dev_to_host_width = 0;
  open(BCXML_FH, $bcxml_file) or acl::Common::mydie("Couldn't open ${bcxml_file} for write! : $!\n");
  while (<BCXML_FH>) {
    my $line = $_;
    if ($line =~ /<INTERFACE port.*chan_id=\S+host/) {
      if ($line =~ /width="(\d+)".*optype="read"/) {
        $host_to_dev_width = $1;
      }
      elsif ($line =~ /width="(\d+)".*optype="write"/) {
        $dev_to_host_width = $1;
      }
      # else error out!
    }
  }
  close BCXML_FH;
  open(TCL, ">$outfile") or acl::Common::mydie("Couldn't open ${outfile} for write! : $!\n");
  print TCL "set INCL_KERNEL_SYSTEM 1\n" if (! $board_qsys);
  print TCL "set USE_SNOOP 1\n" if ($use_snoop);
  print TCL "set USE_CLK2X 1\n" if ($use_clk2x);
  print TCL "set HOST_TO_DEV_READ_WIDTH ${host_to_dev_width}\n";
  print TCL "set DEV_TO_HOST_WRITE_WIDTH ${dev_to_host_width}\n";
  print TCL "set GLB_MEM_AV_WAITREQUEST_ALLOWANCE 6\n" if ($devicefamily =~ /Stratix 10/);
  # Quartus version (Pro or Standard) need to be passed down to tcl simulation script
  my $quartus_pro = $acl::AOCDriverCommon::using_quartus_std ? '0' : '1';
  print TCL "set quartus_pro $quartus_pro\n";
  print TCL "source \"$qsys_generate_system_script\"\n";
  close(TCL);
}

=head2 write_sim_repackage_script($work_dir)

Write out a repackage script for user to simply modify tb or run script without invoking compliation

=cut

sub write_sim_repackage_script($) {
  my $pkg_filepath = shift;
  my $sim_repkg_filename = (acl::Env::is_windows()) ? "sim_repackage.cmd" : "sim_repackage.sh";
  open SIM_REPKG_FH, ">$sim_repkg_filename" or acl::Common::mydie("Couldn't open $sim_repkg_filename for write.\n");
  if (acl::Env::is_linux()) {
    print SIM_REPKG_FH "#!/bin/sh\n\n";
  }
  print SIM_REPKG_FH get_sim_compile_exe() . " > recompile.log\n";
  if (acl::Env::is_linux()) {
    print SIM_REPKG_FH "retval=\$?\n";
    print SIM_REPKG_FH "if [ \${retval} -ne 0 ]; then\n";
    print SIM_REPKG_FH "    echo \"Recompile failed. Check recompile.log for details.\"\n";
    print SIM_REPKG_FH "    exit \${retval}\n";
    print SIM_REPKG_FH "fi\n";
    print SIM_REPKG_FH "cp sim/sys_description.hex .\n";
  } elsif (acl::Env::is_windows()) {
    print SIM_REPKG_FH "set exitCode=%ERRORLEVEL%\n";
    print SIM_REPKG_FH "if %exitCode% NEQ 0 echo \"Recompile failed. Check recompile.log for details.\" && exit /b %exitCode%\n";
    print SIM_REPKG_FH "copy sim\\sys_description.hex .\n";
  }
  my $pkg_string = join(" ", get_sim_package())." ".get_sim_options();
  print SIM_REPKG_FH "aocl binedit fpga-sim.bin package sys_description.hex $pkg_string\n";
  print SIM_REPKG_FH "aocl binedit $pkg_filepath set .acl.fpga.bin fpga-sim.bin\n";
  if (acl::Env::is_windows()) {
    print SIM_REPKG_FH "del fpga-sim.bin sys_description.hex";
  }
  else {
    print SIM_REPKG_FH "rm -f fpga-sim.bin sys_description.hex";
  }
  close SIM_REPKG_FH;
}

=head2 update_sim_debug_depth($fulllog, $less_qsys)

When -reuse-exe flag is used and sim_debug_depth has changed between compiles,
this function updates the simulation run script when the proper waveform log depth
if necessary. Returns 1 if the script was updated, 0 otherwise.

=cut

sub update_sim_debug_depth($$) {
  my ($fulllog, $less_qsys) = @_;
  my $prj_name = acl::File::mybasename($fulllog);
  $prj_name =~ s/\.log$//;

  initialize_simulation_paths($opencl_qsys_name, $acl::AOCDriverCommon::using_quartus_std);
  if ( ! -f $fname_runscript ) {
    acl::Common::mydie("Cannot find run script $fname_runscript\n");
  }

  my $dev_ip_hierarchy_path = "${sim_qsys_name}/${kernel_system_inst}/";
  $dev_ip_hierarchy_path .= ($less_qsys ? 'ks' : "${prj_name}_sys");
  $dev_ip_hierarchy_path .= "/${prj_name}_sys/device_image_ip";
  
  open(RUN_SCRIPT_FILE, "<$fname_runscript") or acl::Common::mydie("Can't open $fname_runscript for read");
  my @lines;
  my $updated = 0;
  while(my $line = <RUN_SCRIPT_FILE>) {
    if ( $line =~ /log -r .*/ ) {
      # This line is the only line that might change with different ghdl setting
      my $new_line = '';
      if (defined($sim_debug_depth)) {
        my $capture_depth = $sim_debug_depth-1;
        $new_line = "log -r -depth $capture_depth ${dev_ip_hierarchy_path}/*\n";
      } else {
        $new_line = "log -r *\n";
      }
      if ( $line eq $new_line ) {
        # This line is the same, just break the loop and don't write to the script.
        last;
      } else {
        # ghdl setting has changed, will write to the script.
        $updated = 1;
      }
      push(@lines, $new_line);
    } else {
      push(@lines, $line);
    }
  }
  close(RUN_SCRIPT_FILE);
  if ($updated == 1) {
    open(RUN_SCRIPT_FILE,">$fname_runscript") or acl::Common::mydie("Can't open $fname_runscript for write");
    foreach my $line (@lines) {
      print RUN_SCRIPT_FILE $line;
    }
    close(RUN_SCRIPT_FILE);
  }
  return $updated;
}

1;
