﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/s3control/S3Control_EXPORTS.h>
#include <aws/s3control/model/S3Tag.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace S3Control {
namespace Model {

/**
 * <p>The container for the Outposts bucket lifecycle rule and
 * operator.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/LifecycleRuleAndOperator">AWS
 * API Reference</a></p>
 */
class LifecycleRuleAndOperator {
 public:
  AWS_S3CONTROL_API LifecycleRuleAndOperator() = default;
  AWS_S3CONTROL_API LifecycleRuleAndOperator(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_S3CONTROL_API LifecycleRuleAndOperator& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_S3CONTROL_API void AddToNode(Aws::Utils::Xml::XmlNode& parentNode) const;

  ///@{
  /**
   * <p>Prefix identifying one or more objects to which the rule applies.</p>
   */
  inline const Aws::String& GetPrefix() const { return m_prefix; }
  inline bool PrefixHasBeenSet() const { return m_prefixHasBeenSet; }
  template <typename PrefixT = Aws::String>
  void SetPrefix(PrefixT&& value) {
    m_prefixHasBeenSet = true;
    m_prefix = std::forward<PrefixT>(value);
  }
  template <typename PrefixT = Aws::String>
  LifecycleRuleAndOperator& WithPrefix(PrefixT&& value) {
    SetPrefix(std::forward<PrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>All of these tags must exist in the object's tag set in order for the rule to
   * apply.</p>
   */
  inline const Aws::Vector<S3Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<S3Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<S3Tag>>
  LifecycleRuleAndOperator& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = S3Tag>
  LifecycleRuleAndOperator& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The non-inclusive minimum object size for the lifecycle rule. Setting this
   * property to 7 means the rule applies to objects with a size that is greater than
   * 7. </p>
   */
  inline long long GetObjectSizeGreaterThan() const { return m_objectSizeGreaterThan; }
  inline bool ObjectSizeGreaterThanHasBeenSet() const { return m_objectSizeGreaterThanHasBeenSet; }
  inline void SetObjectSizeGreaterThan(long long value) {
    m_objectSizeGreaterThanHasBeenSet = true;
    m_objectSizeGreaterThan = value;
  }
  inline LifecycleRuleAndOperator& WithObjectSizeGreaterThan(long long value) {
    SetObjectSizeGreaterThan(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The non-inclusive maximum object size for the lifecycle rule. Setting this
   * property to 77 means the rule applies to objects with a size that is less than
   * 77. </p>
   */
  inline long long GetObjectSizeLessThan() const { return m_objectSizeLessThan; }
  inline bool ObjectSizeLessThanHasBeenSet() const { return m_objectSizeLessThanHasBeenSet; }
  inline void SetObjectSizeLessThan(long long value) {
    m_objectSizeLessThanHasBeenSet = true;
    m_objectSizeLessThan = value;
  }
  inline LifecycleRuleAndOperator& WithObjectSizeLessThan(long long value) {
    SetObjectSizeLessThan(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_prefix;

  Aws::Vector<S3Tag> m_tags;

  long long m_objectSizeGreaterThan{0};

  long long m_objectSizeLessThan{0};
  bool m_prefixHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_objectSizeGreaterThanHasBeenSet = false;
  bool m_objectSizeLessThanHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Control
}  // namespace Aws
