﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm/SSMRequest.h>
#include <aws/ssm/SSM_EXPORTS.h>
#include <aws/ssm/model/InstancePatchStateFilter.h>

#include <utility>

namespace Aws {
namespace SSM {
namespace Model {

/**
 */
class DescribeInstancePatchStatesForPatchGroupRequest : public SSMRequest {
 public:
  AWS_SSM_API DescribeInstancePatchStatesForPatchGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeInstancePatchStatesForPatchGroup"; }

  AWS_SSM_API Aws::String SerializePayload() const override;

  AWS_SSM_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the patch group for which the patch state information should be
   * retrieved.</p>
   */
  inline const Aws::String& GetPatchGroup() const { return m_patchGroup; }
  inline bool PatchGroupHasBeenSet() const { return m_patchGroupHasBeenSet; }
  template <typename PatchGroupT = Aws::String>
  void SetPatchGroup(PatchGroupT&& value) {
    m_patchGroupHasBeenSet = true;
    m_patchGroup = std::forward<PatchGroupT>(value);
  }
  template <typename PatchGroupT = Aws::String>
  DescribeInstancePatchStatesForPatchGroupRequest& WithPatchGroup(PatchGroupT&& value) {
    SetPatchGroup(std::forward<PatchGroupT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Each entry in the array is a structure containing:</p> <ul> <li> <p>Key
   * (string between 1 and 200 characters)</p> </li> <li> <p>Values (array containing
   * a single string)</p> </li> <li> <p>Type (string "Equal", "NotEqual", "LessThan",
   * "GreaterThan")</p> </li> </ul>
   */
  inline const Aws::Vector<InstancePatchStateFilter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<InstancePatchStateFilter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<InstancePatchStateFilter>>
  DescribeInstancePatchStatesForPatchGroupRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = InstancePatchStateFilter>
  DescribeInstancePatchStatesForPatchGroupRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next set of items to return. (You received this token from
   * a previous call.)</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeInstancePatchStatesForPatchGroupRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of patches to return (per page).</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline DescribeInstancePatchStatesForPatchGroupRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_patchGroup;

  Aws::Vector<InstancePatchStateFilter> m_filters;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_patchGroupHasBeenSet = false;
  bool m_filtersHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
