﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/simspaceweaver/SimSpaceWeaverRequest.h>
#include <aws/simspaceweaver/SimSpaceWeaver_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace SimSpaceWeaver {
namespace Model {

/**
 */
class ListAppsRequest : public SimSpaceWeaverRequest {
 public:
  AWS_SIMSPACEWEAVER_API ListAppsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListApps"; }

  AWS_SIMSPACEWEAVER_API Aws::String SerializePayload() const override;

  AWS_SIMSPACEWEAVER_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The name of the domain that you want to list apps for.</p>
   */
  inline const Aws::String& GetDomain() const { return m_domain; }
  inline bool DomainHasBeenSet() const { return m_domainHasBeenSet; }
  template <typename DomainT = Aws::String>
  void SetDomain(DomainT&& value) {
    m_domainHasBeenSet = true;
    m_domain = std::forward<DomainT>(value);
  }
  template <typename DomainT = Aws::String>
  ListAppsRequest& WithDomain(DomainT&& value) {
    SetDomain(std::forward<DomainT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of apps to list.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAppsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If SimSpace Weaver returns <code>nextToken</code>, then there are more
   * results available. The value of <code>nextToken</code> is a unique pagination
   * token for each page. To retrieve the next page, call the operation again using
   * the returned token. Keep all other arguments unchanged. If no results remain,
   * then <code>nextToken</code> is set to <code>null</code>. Each pagination token
   * expires after 24 hours. If you provide a token that isn't valid, then you
   * receive an <i>HTTP 400 ValidationException</i> error.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAppsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the simulation that you want to list apps for.</p>
   */
  inline const Aws::String& GetSimulation() const { return m_simulation; }
  inline bool SimulationHasBeenSet() const { return m_simulationHasBeenSet; }
  template <typename SimulationT = Aws::String>
  void SetSimulation(SimulationT&& value) {
    m_simulationHasBeenSet = true;
    m_simulation = std::forward<SimulationT>(value);
  }
  template <typename SimulationT = Aws::String>
  ListAppsRequest& WithSimulation(SimulationT&& value) {
    SetSimulation(std::forward<SimulationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domain;

  int m_maxResults{0};

  Aws::String m_nextToken;

  Aws::String m_simulation;
  bool m_domainHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_simulationHasBeenSet = false;
};

}  // namespace Model
}  // namespace SimSpaceWeaver
}  // namespace Aws
