﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sesv2/SESV2Request.h>
#include <aws/sesv2/SESV2_EXPORTS.h>
#include <aws/sesv2/model/ListContactsFilter.h>

#include <utility>

namespace Aws {
namespace SESV2 {
namespace Model {

/**
 */
class ListContactsRequest : public SESV2Request {
 public:
  AWS_SESV2_API ListContactsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListContacts"; }

  AWS_SESV2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the contact list.</p>
   */
  inline const Aws::String& GetContactListName() const { return m_contactListName; }
  inline bool ContactListNameHasBeenSet() const { return m_contactListNameHasBeenSet; }
  template <typename ContactListNameT = Aws::String>
  void SetContactListName(ContactListNameT&& value) {
    m_contactListNameHasBeenSet = true;
    m_contactListName = std::forward<ContactListNameT>(value);
  }
  template <typename ContactListNameT = Aws::String>
  ListContactsRequest& WithContactListName(ContactListNameT&& value) {
    SetContactListName(std::forward<ContactListNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that can be applied to a list of contacts.</p>
   */
  inline const ListContactsFilter& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = ListContactsFilter>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = ListContactsFilter>
  ListContactsRequest& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of contacts that may be returned at once, which is dependent on if
   * there are more or less contacts than the value of the PageSize. Use this
   * parameter to paginate results. If additional contacts exist beyond the specified
   * limit, the <code>NextToken</code> element is sent in the response. Use the
   * <code>NextToken</code> value in subsequent requests to retrieve additional
   * contacts.</p>
   */
  inline int GetPageSize() const { return m_pageSize; }
  inline bool PageSizeHasBeenSet() const { return m_pageSizeHasBeenSet; }
  inline void SetPageSize(int value) {
    m_pageSizeHasBeenSet = true;
    m_pageSize = value;
  }
  inline ListContactsRequest& WithPageSize(int value) {
    SetPageSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A string token indicating that there might be additional contacts available
   * to be listed. Use the token provided in the Response to use in the subsequent
   * call to ListContacts with the same parameters to retrieve the next page of
   * contacts.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListContactsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_contactListName;

  ListContactsFilter m_filter;

  int m_pageSize{0};

  Aws::String m_nextToken;
  bool m_contactListNameHasBeenSet = false;
  bool m_filterHasBeenSet = false;
  bool m_pageSizeHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace SESV2
}  // namespace Aws
