﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/servicecatalog/ServiceCatalogRequest.h>
#include <aws/servicecatalog/ServiceCatalog_EXPORTS.h>
#include <aws/servicecatalog/model/DescribePortfolioShareType.h>

#include <utility>

namespace Aws {
namespace ServiceCatalog {
namespace Model {

/**
 */
class DescribePortfolioSharesRequest : public ServiceCatalogRequest {
 public:
  AWS_SERVICECATALOG_API DescribePortfolioSharesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribePortfolioShares"; }

  AWS_SERVICECATALOG_API Aws::String SerializePayload() const override;

  AWS_SERVICECATALOG_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The unique identifier of the portfolio for which shares will be
   * retrieved.</p>
   */
  inline const Aws::String& GetPortfolioId() const { return m_portfolioId; }
  inline bool PortfolioIdHasBeenSet() const { return m_portfolioIdHasBeenSet; }
  template <typename PortfolioIdT = Aws::String>
  void SetPortfolioId(PortfolioIdT&& value) {
    m_portfolioIdHasBeenSet = true;
    m_portfolioId = std::forward<PortfolioIdT>(value);
  }
  template <typename PortfolioIdT = Aws::String>
  DescribePortfolioSharesRequest& WithPortfolioId(PortfolioIdT&& value) {
    SetPortfolioId(std::forward<PortfolioIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of portfolio share to summarize. This field acts as a filter on the
   * type of portfolio share, which can be one of the following:</p> <p>1.
   * <code>ACCOUNT</code> - Represents an external account to account share.</p>
   * <p>2. <code>ORGANIZATION</code> - Represents a share to an organization. This
   * share is available to every account in the organization.</p> <p>3.
   * <code>ORGANIZATIONAL_UNIT</code> - Represents a share to an organizational
   * unit.</p> <p>4. <code>ORGANIZATION_MEMBER_ACCOUNT</code> - Represents a share to
   * an account in the organization.</p>
   */
  inline DescribePortfolioShareType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(DescribePortfolioShareType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline DescribePortfolioSharesRequest& WithType(DescribePortfolioShareType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The page token for the next set of results. To retrieve the first set of
   * results, use null.</p>
   */
  inline const Aws::String& GetPageToken() const { return m_pageToken; }
  inline bool PageTokenHasBeenSet() const { return m_pageTokenHasBeenSet; }
  template <typename PageTokenT = Aws::String>
  void SetPageToken(PageTokenT&& value) {
    m_pageTokenHasBeenSet = true;
    m_pageToken = std::forward<PageTokenT>(value);
  }
  template <typename PageTokenT = Aws::String>
  DescribePortfolioSharesRequest& WithPageToken(PageTokenT&& value) {
    SetPageToken(std::forward<PageTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items to return with this call.</p>
   */
  inline int GetPageSize() const { return m_pageSize; }
  inline bool PageSizeHasBeenSet() const { return m_pageSizeHasBeenSet; }
  inline void SetPageSize(int value) {
    m_pageSizeHasBeenSet = true;
    m_pageSize = value;
  }
  inline DescribePortfolioSharesRequest& WithPageSize(int value) {
    SetPageSize(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_portfolioId;

  DescribePortfolioShareType m_type{DescribePortfolioShareType::NOT_SET};

  Aws::String m_pageToken;

  int m_pageSize{0};
  bool m_portfolioIdHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_pageTokenHasBeenSet = false;
  bool m_pageSizeHasBeenSet = false;
};

}  // namespace Model
}  // namespace ServiceCatalog
}  // namespace Aws
