﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class CreatePresignedMlflowTrackingServerUrlRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API CreatePresignedMlflowTrackingServerUrlRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreatePresignedMlflowTrackingServerUrl"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the tracking server to connect to your MLflow UI.</p>
   */
  inline const Aws::String& GetTrackingServerName() const { return m_trackingServerName; }
  inline bool TrackingServerNameHasBeenSet() const { return m_trackingServerNameHasBeenSet; }
  template <typename TrackingServerNameT = Aws::String>
  void SetTrackingServerName(TrackingServerNameT&& value) {
    m_trackingServerNameHasBeenSet = true;
    m_trackingServerName = std::forward<TrackingServerNameT>(value);
  }
  template <typename TrackingServerNameT = Aws::String>
  CreatePresignedMlflowTrackingServerUrlRequest& WithTrackingServerName(TrackingServerNameT&& value) {
    SetTrackingServerName(std::forward<TrackingServerNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The duration in seconds that your presigned URL is valid. The presigned URL
   * can be used only once.</p>
   */
  inline int GetExpiresInSeconds() const { return m_expiresInSeconds; }
  inline bool ExpiresInSecondsHasBeenSet() const { return m_expiresInSecondsHasBeenSet; }
  inline void SetExpiresInSeconds(int value) {
    m_expiresInSecondsHasBeenSet = true;
    m_expiresInSeconds = value;
  }
  inline CreatePresignedMlflowTrackingServerUrlRequest& WithExpiresInSeconds(int value) {
    SetExpiresInSeconds(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The duration in seconds that your MLflow UI session is valid.</p>
   */
  inline int GetSessionExpirationDurationInSeconds() const { return m_sessionExpirationDurationInSeconds; }
  inline bool SessionExpirationDurationInSecondsHasBeenSet() const { return m_sessionExpirationDurationInSecondsHasBeenSet; }
  inline void SetSessionExpirationDurationInSeconds(int value) {
    m_sessionExpirationDurationInSecondsHasBeenSet = true;
    m_sessionExpirationDurationInSeconds = value;
  }
  inline CreatePresignedMlflowTrackingServerUrlRequest& WithSessionExpirationDurationInSeconds(int value) {
    SetSessionExpirationDurationInSeconds(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_trackingServerName;

  int m_expiresInSeconds{0};

  int m_sessionExpirationDurationInSeconds{0};
  bool m_trackingServerNameHasBeenSet = false;
  bool m_expiresInSecondsHasBeenSet = false;
  bool m_sessionExpirationDurationInSecondsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
