﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/mediapackagev2/Mediapackagev2_EXPORTS.h>
#include <aws/mediapackagev2/model/Destination.h>
#include <aws/mediapackagev2/model/HarvestJobStatus.h>
#include <aws/mediapackagev2/model/HarvestedManifests.h>
#include <aws/mediapackagev2/model/HarvesterScheduleConfiguration.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace mediapackagev2 {
namespace Model {

/**
 * <p>Represents a harvest job resource in MediaPackage v2, which is used to export
 * content from an origin endpoint to an S3 bucket.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/mediapackagev2-2022-12-25/HarvestJob">AWS
 * API Reference</a></p>
 */
class HarvestJob {
 public:
  AWS_MEDIAPACKAGEV2_API HarvestJob() = default;
  AWS_MEDIAPACKAGEV2_API HarvestJob(Aws::Utils::Json::JsonView jsonValue);
  AWS_MEDIAPACKAGEV2_API HarvestJob& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_MEDIAPACKAGEV2_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the channel group containing the channel associated with this
   * harvest job.</p>
   */
  inline const Aws::String& GetChannelGroupName() const { return m_channelGroupName; }
  inline bool ChannelGroupNameHasBeenSet() const { return m_channelGroupNameHasBeenSet; }
  template <typename ChannelGroupNameT = Aws::String>
  void SetChannelGroupName(ChannelGroupNameT&& value) {
    m_channelGroupNameHasBeenSet = true;
    m_channelGroupName = std::forward<ChannelGroupNameT>(value);
  }
  template <typename ChannelGroupNameT = Aws::String>
  HarvestJob& WithChannelGroupName(ChannelGroupNameT&& value) {
    SetChannelGroupName(std::forward<ChannelGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the channel associated with this harvest job.</p>
   */
  inline const Aws::String& GetChannelName() const { return m_channelName; }
  inline bool ChannelNameHasBeenSet() const { return m_channelNameHasBeenSet; }
  template <typename ChannelNameT = Aws::String>
  void SetChannelName(ChannelNameT&& value) {
    m_channelNameHasBeenSet = true;
    m_channelName = std::forward<ChannelNameT>(value);
  }
  template <typename ChannelNameT = Aws::String>
  HarvestJob& WithChannelName(ChannelNameT&& value) {
    SetChannelName(std::forward<ChannelNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the origin endpoint associated with this harvest job.</p>
   */
  inline const Aws::String& GetOriginEndpointName() const { return m_originEndpointName; }
  inline bool OriginEndpointNameHasBeenSet() const { return m_originEndpointNameHasBeenSet; }
  template <typename OriginEndpointNameT = Aws::String>
  void SetOriginEndpointName(OriginEndpointNameT&& value) {
    m_originEndpointNameHasBeenSet = true;
    m_originEndpointName = std::forward<OriginEndpointNameT>(value);
  }
  template <typename OriginEndpointNameT = Aws::String>
  HarvestJob& WithOriginEndpointName(OriginEndpointNameT&& value) {
    SetOriginEndpointName(std::forward<OriginEndpointNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The S3 destination where the harvested content will be placed.</p>
   */
  inline const Destination& GetDestination() const { return m_destination; }
  inline bool DestinationHasBeenSet() const { return m_destinationHasBeenSet; }
  template <typename DestinationT = Destination>
  void SetDestination(DestinationT&& value) {
    m_destinationHasBeenSet = true;
    m_destination = std::forward<DestinationT>(value);
  }
  template <typename DestinationT = Destination>
  HarvestJob& WithDestination(DestinationT&& value) {
    SetDestination(std::forward<DestinationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the harvest job.</p>
   */
  inline const Aws::String& GetHarvestJobName() const { return m_harvestJobName; }
  inline bool HarvestJobNameHasBeenSet() const { return m_harvestJobNameHasBeenSet; }
  template <typename HarvestJobNameT = Aws::String>
  void SetHarvestJobName(HarvestJobNameT&& value) {
    m_harvestJobNameHasBeenSet = true;
    m_harvestJobName = std::forward<HarvestJobNameT>(value);
  }
  template <typename HarvestJobNameT = Aws::String>
  HarvestJob& WithHarvestJobName(HarvestJobNameT&& value) {
    SetHarvestJobName(std::forward<HarvestJobNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of manifests that are being or have been harvested.</p>
   */
  inline const HarvestedManifests& GetHarvestedManifests() const { return m_harvestedManifests; }
  inline bool HarvestedManifestsHasBeenSet() const { return m_harvestedManifestsHasBeenSet; }
  template <typename HarvestedManifestsT = HarvestedManifests>
  void SetHarvestedManifests(HarvestedManifestsT&& value) {
    m_harvestedManifestsHasBeenSet = true;
    m_harvestedManifests = std::forward<HarvestedManifestsT>(value);
  }
  template <typename HarvestedManifestsT = HarvestedManifests>
  HarvestJob& WithHarvestedManifests(HarvestedManifestsT&& value) {
    SetHarvestedManifests(std::forward<HarvestedManifestsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional description of the harvest job.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  HarvestJob& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration for when the harvest job is scheduled to run.</p>
   */
  inline const HarvesterScheduleConfiguration& GetScheduleConfiguration() const { return m_scheduleConfiguration; }
  inline bool ScheduleConfigurationHasBeenSet() const { return m_scheduleConfigurationHasBeenSet; }
  template <typename ScheduleConfigurationT = HarvesterScheduleConfiguration>
  void SetScheduleConfiguration(ScheduleConfigurationT&& value) {
    m_scheduleConfigurationHasBeenSet = true;
    m_scheduleConfiguration = std::forward<ScheduleConfigurationT>(value);
  }
  template <typename ScheduleConfigurationT = HarvesterScheduleConfiguration>
  HarvestJob& WithScheduleConfiguration(ScheduleConfigurationT&& value) {
    SetScheduleConfiguration(std::forward<ScheduleConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the harvest job.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  HarvestJob& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time when the harvest job was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedAt() const { return m_createdAt; }
  inline bool CreatedAtHasBeenSet() const { return m_createdAtHasBeenSet; }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  void SetCreatedAt(CreatedAtT&& value) {
    m_createdAtHasBeenSet = true;
    m_createdAt = std::forward<CreatedAtT>(value);
  }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  HarvestJob& WithCreatedAt(CreatedAtT&& value) {
    SetCreatedAt(std::forward<CreatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time when the harvest job was last modified.</p>
   */
  inline const Aws::Utils::DateTime& GetModifiedAt() const { return m_modifiedAt; }
  inline bool ModifiedAtHasBeenSet() const { return m_modifiedAtHasBeenSet; }
  template <typename ModifiedAtT = Aws::Utils::DateTime>
  void SetModifiedAt(ModifiedAtT&& value) {
    m_modifiedAtHasBeenSet = true;
    m_modifiedAt = std::forward<ModifiedAtT>(value);
  }
  template <typename ModifiedAtT = Aws::Utils::DateTime>
  HarvestJob& WithModifiedAt(ModifiedAtT&& value) {
    SetModifiedAt(std::forward<ModifiedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current status of the harvest job (e.g., QUEUED, IN_PROGRESS, CANCELLED,
   * COMPLETED, FAILED).</p>
   */
  inline HarvestJobStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(HarvestJobStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline HarvestJob& WithStatus(HarvestJobStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An error message if the harvest job encountered any issues.</p>
   */
  inline const Aws::String& GetErrorMessage() const { return m_errorMessage; }
  inline bool ErrorMessageHasBeenSet() const { return m_errorMessageHasBeenSet; }
  template <typename ErrorMessageT = Aws::String>
  void SetErrorMessage(ErrorMessageT&& value) {
    m_errorMessageHasBeenSet = true;
    m_errorMessage = std::forward<ErrorMessageT>(value);
  }
  template <typename ErrorMessageT = Aws::String>
  HarvestJob& WithErrorMessage(ErrorMessageT&& value) {
    SetErrorMessage(std::forward<ErrorMessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current version of the harvest job. Used for concurrency control.</p>
   */
  inline const Aws::String& GetETag() const { return m_eTag; }
  inline bool ETagHasBeenSet() const { return m_eTagHasBeenSet; }
  template <typename ETagT = Aws::String>
  void SetETag(ETagT&& value) {
    m_eTagHasBeenSet = true;
    m_eTag = std::forward<ETagT>(value);
  }
  template <typename ETagT = Aws::String>
  HarvestJob& WithETag(ETagT&& value) {
    SetETag(std::forward<ETagT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_channelGroupName;

  Aws::String m_channelName;

  Aws::String m_originEndpointName;

  Destination m_destination;

  Aws::String m_harvestJobName;

  HarvestedManifests m_harvestedManifests;

  Aws::String m_description;

  HarvesterScheduleConfiguration m_scheduleConfiguration;

  Aws::String m_arn;

  Aws::Utils::DateTime m_createdAt{};

  Aws::Utils::DateTime m_modifiedAt{};

  HarvestJobStatus m_status{HarvestJobStatus::NOT_SET};

  Aws::String m_errorMessage;

  Aws::String m_eTag;
  bool m_channelGroupNameHasBeenSet = false;
  bool m_channelNameHasBeenSet = false;
  bool m_originEndpointNameHasBeenSet = false;
  bool m_destinationHasBeenSet = false;
  bool m_harvestJobNameHasBeenSet = false;
  bool m_harvestedManifestsHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_scheduleConfigurationHasBeenSet = false;
  bool m_arnHasBeenSet = false;
  bool m_createdAtHasBeenSet = false;
  bool m_modifiedAtHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_errorMessageHasBeenSet = false;
  bool m_eTagHasBeenSet = false;
};

}  // namespace Model
}  // namespace mediapackagev2
}  // namespace Aws
