﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/elasticfilesystem/EFSRequest.h>
#include <aws/elasticfilesystem/EFS_EXPORTS.h>
#include <aws/elasticfilesystem/model/ThroughputMode.h>

#include <utility>

namespace Aws {
namespace EFS {
namespace Model {

/**
 */
class UpdateFileSystemRequest : public EFSRequest {
 public:
  AWS_EFS_API UpdateFileSystemRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateFileSystem"; }

  AWS_EFS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the file system that you want to update.</p>
   */
  inline const Aws::String& GetFileSystemId() const { return m_fileSystemId; }
  inline bool FileSystemIdHasBeenSet() const { return m_fileSystemIdHasBeenSet; }
  template <typename FileSystemIdT = Aws::String>
  void SetFileSystemId(FileSystemIdT&& value) {
    m_fileSystemIdHasBeenSet = true;
    m_fileSystemId = std::forward<FileSystemIdT>(value);
  }
  template <typename FileSystemIdT = Aws::String>
  UpdateFileSystemRequest& WithFileSystemId(FileSystemIdT&& value) {
    SetFileSystemId(std::forward<FileSystemIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>(Optional) Updates the file system's throughput mode. If you're not updating
   * your throughput mode, you don't need to provide this value in your request. If
   * you are changing the <code>ThroughputMode</code> to <code>provisioned</code>,
   * you must also set a value for <code>ProvisionedThroughputInMibps</code>.</p>
   */
  inline ThroughputMode GetThroughputMode() const { return m_throughputMode; }
  inline bool ThroughputModeHasBeenSet() const { return m_throughputModeHasBeenSet; }
  inline void SetThroughputMode(ThroughputMode value) {
    m_throughputModeHasBeenSet = true;
    m_throughputMode = value;
  }
  inline UpdateFileSystemRequest& WithThroughputMode(ThroughputMode value) {
    SetThroughputMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>(Optional) The throughput, measured in mebibytes per second (MiBps), that you
   * want to provision for a file system that you're creating. Required if
   * <code>ThroughputMode</code> is set to <code>provisioned</code>. Valid values are
   * 1-3414 MiBps, with the upper limit depending on Region. To increase this limit,
   * contact Amazon Web ServicesSupport. For more information, see <a
   * href="https://docs.aws.amazon.com/efs/latest/ug/limits.html#soft-limits">Amazon
   * EFS quotas that you can increase</a> in the <i>Amazon EFS User Guide</i>.</p>
   */
  inline double GetProvisionedThroughputInMibps() const { return m_provisionedThroughputInMibps; }
  inline bool ProvisionedThroughputInMibpsHasBeenSet() const { return m_provisionedThroughputInMibpsHasBeenSet; }
  inline void SetProvisionedThroughputInMibps(double value) {
    m_provisionedThroughputInMibpsHasBeenSet = true;
    m_provisionedThroughputInMibps = value;
  }
  inline UpdateFileSystemRequest& WithProvisionedThroughputInMibps(double value) {
    SetProvisionedThroughputInMibps(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_fileSystemId;

  ThroughputMode m_throughputMode{ThroughputMode::NOT_SET};

  double m_provisionedThroughputInMibps{0.0};
  bool m_fileSystemIdHasBeenSet = false;
  bool m_throughputModeHasBeenSet = false;
  bool m_provisionedThroughputInMibpsHasBeenSet = false;
};

}  // namespace Model
}  // namespace EFS
}  // namespace Aws
