﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/autoscaling/AutoScalingRequest.h>
#include <aws/autoscaling/AutoScaling_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace AutoScaling {
namespace Model {

/**
 */
class AttachLoadBalancersRequest : public AutoScalingRequest {
 public:
  AWS_AUTOSCALING_API AttachLoadBalancersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AttachLoadBalancers"; }

  AWS_AUTOSCALING_API Aws::String SerializePayload() const override;

 protected:
  AWS_AUTOSCALING_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the Auto Scaling group.</p>
   */
  inline const Aws::String& GetAutoScalingGroupName() const { return m_autoScalingGroupName; }
  inline bool AutoScalingGroupNameHasBeenSet() const { return m_autoScalingGroupNameHasBeenSet; }
  template <typename AutoScalingGroupNameT = Aws::String>
  void SetAutoScalingGroupName(AutoScalingGroupNameT&& value) {
    m_autoScalingGroupNameHasBeenSet = true;
    m_autoScalingGroupName = std::forward<AutoScalingGroupNameT>(value);
  }
  template <typename AutoScalingGroupNameT = Aws::String>
  AttachLoadBalancersRequest& WithAutoScalingGroupName(AutoScalingGroupNameT&& value) {
    SetAutoScalingGroupName(std::forward<AutoScalingGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The names of the load balancers. You can specify up to 10 load balancers.</p>
   */
  inline const Aws::Vector<Aws::String>& GetLoadBalancerNames() const { return m_loadBalancerNames; }
  inline bool LoadBalancerNamesHasBeenSet() const { return m_loadBalancerNamesHasBeenSet; }
  template <typename LoadBalancerNamesT = Aws::Vector<Aws::String>>
  void SetLoadBalancerNames(LoadBalancerNamesT&& value) {
    m_loadBalancerNamesHasBeenSet = true;
    m_loadBalancerNames = std::forward<LoadBalancerNamesT>(value);
  }
  template <typename LoadBalancerNamesT = Aws::Vector<Aws::String>>
  AttachLoadBalancersRequest& WithLoadBalancerNames(LoadBalancerNamesT&& value) {
    SetLoadBalancerNames(std::forward<LoadBalancerNamesT>(value));
    return *this;
  }
  template <typename LoadBalancerNamesT = Aws::String>
  AttachLoadBalancersRequest& AddLoadBalancerNames(LoadBalancerNamesT&& value) {
    m_loadBalancerNamesHasBeenSet = true;
    m_loadBalancerNames.emplace_back(std::forward<LoadBalancerNamesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_autoScalingGroupName;

  Aws::Vector<Aws::String> m_loadBalancerNames;
  bool m_autoScalingGroupNameHasBeenSet = false;
  bool m_loadBalancerNamesHasBeenSet = false;
};

}  // namespace Model
}  // namespace AutoScaling
}  // namespace Aws
